# copyright 2015

#    under the License.

"""
Backup driver for huawei eBackup

"""

import collections
import copy
import functools
import json
import time
import traceback
import uuid
import threading
import os
from cinder.backup.driver import BackupDriverWithVerify
from oslo_utils import timeutils

from cinder import exception
from cinder.backup.drivers.ebackupagent import convert_error, call_db_fun, Connection, EVS, \
    Utils, is_dj_env, FSPythonApi, NovaApi
from cinder.backup.drivers.ebackupconst import Constants as cst
from cinder.backup.drivers.ebackupconst import ErrorCode
from cinder.volume import utils as volume_utils
from cinder.backup.lib.ebackuprestful import EbackupConnectionClient
from cinder.backup.lib.Uranus import Uranus

try:
    # M version
    from oslo_config import cfg
    from oslo_log import log as logging
    from oslo_utils import excutils
    from cinder import objects
except:
    # J version
    from oslo.config import cfg
    from cinder.openstack.common import log as logging
    from cinder.openstack.common import excutils

from oslo_utils import importutils
from six.moves import configparser as config_parser

profiler = importutils.try_import("osprofiler.profiler")
osprofiler_web = importutils.try_import('osprofiler.web')

ENV_PUBLIC_CLOUD = '1'
ENV_PRIVATE_CLOUD = '2'
ENV_HYBRID_CLOUD = '3'
UNIT_S3 = '1'
UNIT_NAS = '2'
UNIT_SAN = '3'
FILESYS_CIFS = '1'
FILESYS_NFS = '2'
FILESYS_SAN = '3'
FILESYS_S3 = '7'
IMAGE_OBJ_ID = '57370'
COPY_PLAN_OBJ_ID = '57378'
BACKUP_MODE_LOCAL = 0
BACKUP_MODE_TO_CLOUD = 1
SELECTION_POLICY_NAS_PREFERENTIALLY = '1'
POLICY_SCHEDULE_TYPE_ONCE = 2
SNAPSHOT_BACKUP_MACHINE = 1
SNAPSHOT_BACKUP_SINGLE_VOL = 0
REMOTE_REPLICATION_POP_KEY = ('SrcRegionID', 'EBKIP', 'EBKPORT', 'IAMUser', 'IAMPassword', 'SrcTenantID', 'ODL', 'AK',
                              'SK', 'DEK', 'OVK', 'OCMKID', 'DomainID', 'NeedAccelerate', 'OBPID', 'PR', 'BandWidth',
                              'TAG')

# file path
file_abs_path = os.path.dirname(os.path.abspath(__file__))
ebackup_cfg_file = str(file_abs_path) + "/ebackup.conf"

ebackup_server_opts = [
    cfg.StrOpt('ebackup_server_ip',
               default='',
               help='ebackup server ip'),
    cfg.StrOpt('ebackup_server_port',
               default='8088',
               help='eBackup server port'),
    cfg.StrOpt('ebackup_login_username',
               default='username',
               help='eBackup login username'),
    cfg.StrOpt('ebackup_login_password_v2',
               default='',
               help='eBackup login password_v2'),
    cfg.StrOpt('ebackup_login_crt',
               default='',
               help=''),
    cfg.ListOpt('azs',
                default='',
                help=''),
    cfg.StrOpt('fusion_storage_ip',
               default='default',
               help=''),
    cfg.StrOpt('physical_network_type',
               default='',
               help=''),
    cfg.StrOpt('fusion_storage_agent_ip',
               default='',
               help=''),
    cfg.StrOpt('uuid',
               default='',
               help=''),
    cfg.StrOpt('extend',
               default=False,
               help=''),
    cfg.StrOpt('tenant_white_list',
               default='',
               help='')
]

hws_opts = [
    cfg.ListOpt('ebackup_servers',
                default='',
                help='ebackup servers'),
    cfg.StrOpt('ebackup_env_type',
               default='',
               help=''),
    cfg.StrOpt('ebackup_uds_data_layout',
               default='2',
               help=''),
    cfg.StrOpt('ebackup_storage_unit_path',
               default='',
               help=''),
    cfg.StrOpt('ebackup_storage_password_v2',
               default='',
               help=''),
    cfg.StrOpt('ebackup_storage_username_v2',
               default='',
               help=''),
    cfg.StrOpt('ebackup_storage_s3_protocol',
               default='',
               help=''),
    cfg.StrOpt('ebackup_bucket_name',
               default='',
               help=''),
    cfg.StrOpt('ebackup_cifs_data_layout',
               default='2',
               help=''),
    cfg.StrOpt('ebackup_nfs_data_layout',
               default='2',
               help=''),
    cfg.StrOpt('ebackup_san_data_layout',
               default='2',
               help=''),
    cfg.StrOpt('ebackup_storage_selection_policy',
               default='',
               help=''),
    cfg.StrOpt('ebackup_storage_s3_protocol',
               default='',
               help=''),
    cfg.StrOpt('ebackup_nova_endpoint',
               default='',
               help=''),
    cfg.StrOpt('ebackup_nova_ecs_api',
               default='',
               help=''),
    cfg.StrOpt('ebackup_task_retry_count',
               default='0',
               help=''),
    cfg.StrOpt('ebackup_task_retry_time_interval',
               default='300',
               help=''),
    cfg.StrOpt('ebackup_lazyloading',
               default='0',
               help=''),
    cfg.BoolOpt('ebackup_vm_sdr',
                default=False,
                help=''),
    cfg.BoolOpt('ebackup_sync',
                default=False,
                help='')
]
CINDER_GROUP = cfg.OptGroup(name='eBackup', title='eBackup group')
CINDER_CONF = cfg.CONF
CINDER_CONF.register_opts(hws_opts)
CINDER_CONF.register_opts(ebackup_server_opts)
ebackup_driver_conf = None

# keylist decrypt
try:
    try:
        from FSSecurity import crypt as fscrypt
    except ImportError:
        from FSComponentUtil import crypt as fscrypt
except ImportError:
    if not is_dj_env():
        exit("quit program.")

LOG = logging.getLogger(__name__)


def get_conf_option(conf, option):
    option_value = ""
    if conf.has_option("config", option):
        option_value = conf.get("config", option)
    return option_value


class backup_config(object):
    ebackup_env_type = ''
    ebackup_uds_data_layout = ''
    ebackup_storage_unit_path = ''
    ebackup_storage_password = ''
    ebackup_storage_username = ''
    ebackup_storage_s3_protocol = ''
    ebackup_bucket_name = ''
    ebackup_cifs_data_layout = ''
    ebackup_nfs_data_layout = ''
    ebackup_san_data_layout = ''
    ebackup_storage_selection_policy = ''
    ebackup_task_retry_count = '0'
    ebackup_task_retry_time_interval = '0'
    ebackup_lazyloading = '0'
    ebackup_nova_endpoint = ''
    ebackup_nova_ecs_api = ''
    ebackup_vm_sdr = False
    ebackup_sync_controller = True

    def init_conf(self):
        if not os.path.exists('/etc/huawei/dj/cfg/sys.ini'):
            self.ebackup_env_type = CINDER_CONF.ebackup_env_type
            self.ebackup_sync_controller = CINDER_CONF.ebackup_sync
            if self.ebackup_env_type == ENV_PUBLIC_CLOUD:
                self.ebackup_uds_data_layout = CINDER_CONF.ebackup_uds_data_layout
                self.ebackup_storage_unit_path = CINDER_CONF.ebackup_storage_unit_path
                self.ebackup_storage_password = CINDER_CONF.ebackup_storage_password_v2
                self.ebackup_storage_username = CINDER_CONF.ebackup_storage_username_v2
                self.ebackup_storage_s3_protocol = CINDER_CONF.ebackup_storage_s3_protocol
                self.ebackup_bucket_name = CINDER_CONF.ebackup_bucket_name
                self.ebackup_lazyloading = CINDER_CONF.ebackup_lazyloading
                self.ebackup_vm_sdr = CINDER_CONF.ebackup_vm_sdr
                self.ebackup_nova_ecs_api = CINDER_CONF.ebackup_nova_ecs_api
                self.ebackup_nova_endpoint = CINDER_CONF.ebackup_nova_endpoint
                try:
                    self.ebackup_task_retry_count = CINDER_CONF.ebackup_task_retry_count
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_count = '0'
                try:
                    self.ebackup_task_retry_time_interval = CINDER_CONF.ebackup_task_retry_time_interval
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_time_interval = '0'
            elif self.ebackup_env_type == ENV_PRIVATE_CLOUD:
                self.ebackup_cifs_data_layout = CINDER_CONF.ebackup_cifs_data_layout
                self.ebackup_nfs_data_layout = CINDER_CONF.ebackup_nfs_data_layout
                self.ebackup_uds_data_layout = CINDER_CONF.ebackup_uds_data_layout
                self.ebackup_storage_selection_policy = CINDER_CONF.ebackup_storage_selection_policy
                try:
                    self.ebackup_task_retry_count = CINDER_CONF.ebackup_task_retry_count
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_count = '0'
                try:
                    self.ebackup_task_retry_time_interval = CINDER_CONF.ebackup_task_retry_time_interval
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_time_interval = '0'
            elif self.ebackup_env_type == ENV_HYBRID_CLOUD:
                self.ebackup_cifs_data_layout = CINDER_CONF.ebackup_cifs_data_layout
                self.ebackup_nfs_data_layout = CINDER_CONF.ebackup_nfs_data_layout
                self.ebackup_san_data_layout = CINDER_CONF.ebackup_san_data_layout
                self.ebackup_uds_data_layout = CINDER_CONF.ebackup_uds_data_layout
        else:
            cfg = config_parser.ConfigParser()
            cfg.read(ebackup_cfg_file)
            self.ebackup_env_type = cfg.get("config", "ebackup_env_type")
            if self.ebackup_env_type == ENV_PUBLIC_CLOUD:
                self.ebackup_uds_data_layout = cfg.get("config", "ebackup_uds_data_layout")
                self.ebackup_storage_unit_path = cfg.get("config", "ebackup_storage_unit_path")
                self.ebackup_storage_password = cfg.get("config", "ebackup_storage_password_v2")
                self.ebackup_storage_username = cfg.get("config", "ebackup_storage_username_v2")
                self.ebackup_storage_s3_protocol = cfg.get("config", "ebackup_storage_s3_protocol")
                self.ebackup_bucket_name = cfg.get("config", "ebackup_bucket_name")
                try:
                    self.ebackup_task_retry_count = cfg.get("config", "ebackup_task_retry_count")
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_count = '0'
                try:
                    self.ebackup_task_retry_time_interval = cfg.get("config", "ebackup_task_retry_time_interval")
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_time_interval = '0'
            elif self.ebackup_env_type == ENV_PRIVATE_CLOUD:
                self.ebackup_cifs_data_layout = cfg.get("config", "ebackup_cifs_data_layout")
                self.ebackup_nfs_data_layout = cfg.get("config", "ebackup_nfs_data_layout")
                try:
                    self.ebackup_uds_data_layout = cfg.get("config", "ebackup_uds_data_layout")
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_uds_data_layout = ""
                try:
                    self.ebackup_storage_selection_policy = cfg.get("config", "ebackup_storage_selection_policy")
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_storage_selection_policy = SELECTION_POLICY_NAS_PREFERENTIALLY
                try:
                    self.ebackup_task_retry_count = cfg.get("config", "ebackup_task_retry_count")
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_count = '0'
                try:
                    self.ebackup_task_retry_time_interval = cfg.get("config", "ebackup_task_retry_time_interval")
                except Exception as msg:
                    LOG.info(msg)
                    self.ebackup_task_retry_time_interval = '0'
            elif self.ebackup_env_type == ENV_HYBRID_CLOUD:
                self.ebackup_cifs_data_layout = get_conf_option(cfg, "ebackup_cifs_data_layout")
                self.ebackup_nfs_data_layout = get_conf_option(cfg, "ebackup_nfs_data_layout")
                self.ebackup_san_data_layout = get_conf_option(cfg, "ebackup_san_data_layout")
                self.ebackup_uds_data_layout = get_conf_option(cfg, "ebackup_uds_data_layout")
        LOG.debug("backup_config: ebackup_task_retry_count(%s), ebackup_task_retry_time_interval(%s)." % (
            self.ebackup_task_retry_count, self.ebackup_task_retry_time_interval))


class ebackup_server_config(object):
    ebackup_servers = []
    ebackup_azs = dict()
    ebackup_server_ip = ''
    ebackup_server_port = ''
    ebackup_login_username = ''
    ebackup_login_password = ''
    ebackup_login_crt = ''
    fusion_storage_ip = ''
    physical_network_type = ''
    fusion_storage_agent_ip = ''
    uuid = ''
    extend = False
    tenant_write_list = list()

    def init_conf(self):
        if not os.path.exists('/etc/huawei/dj/cfg/sys.ini'):
            self.ebackup_server_ip = CINDER_CONF.ebackup_server_ip
            self.ebackup_server_port = CINDER_CONF.ebackup_server_port
            self.ebackup_login_username = CINDER_CONF.ebackup_login_username
            self.ebackup_login_password = CINDER_CONF.ebackup_login_password_v2
            self.ebackup_login_crt = CINDER_CONF.ebackup_login_crt
            try:
                self.ebackup_servers = CINDER_CONF.ebackup_servers
                for server in self.ebackup_servers:
                    CINDER_GROUP.name = server
                    CINDER_CONF.register_group(CINDER_GROUP)
                    CINDER_CONF.register_opts(ebackup_server_opts, CINDER_GROUP)
                    ebk_s_info = dict()
                    ebk_s_info["ebackup_server_ip"] = self.__get_conf_val(server, 'ebackup_server_ip')
                    ebk_s_info["ebackup_server_port"] = self.__get_conf_val(server, 'ebackup_server_port')
                    ebk_s_info["ebackup_login_username"] = self.__get_conf_val(server, 'ebackup_login_username')
                    ebk_s_info["ebackup_login_password"] = self.__get_conf_val(server, 'ebackup_login_password_v2')
                    ebk_s_info["ebackup_login_crt"] = self.__get_conf_val(server, 'ebackup_login_crt')
                    ebk_s_info["fusion_storage_ip"] = self.__get_conf_val(server, 'fusion_storage_ip')
                    ebk_s_info["physical_network_type"] = self.__get_conf_val(server, 'physical_network_type')
                    ebk_s_info["fusion_storage_agent_ip"] = self.__get_conf_val(server, 'fusion_storage_agent_ip')
                    ebk_s_info['uuid'] = self.__get_conf_val(server, 'uuid')
                    ebk_s_info['extend'] = self.__get_conf_val(server, 'extend')
                    ebk_s_info['tenant_white_list'] = self.__get_conf_val(server, 'tenant_white_list')
                    azs = self.__get_conf_val(server, 'azs')
                    for az in azs:
                        key = (az, ebk_s_info["fusion_storage_ip"])
                        if not self.ebackup_azs.keys() or key not in self.ebackup_azs.keys():
                            self.ebackup_azs[key] = list()
                        self.ebackup_azs[key].append(copy.deepcopy(ebk_s_info))
                        key_az_pnt = (az, ebk_s_info["physical_network_type"])
                        if not self.ebackup_azs.keys() or key_az_pnt not in self.ebackup_azs.keys():
                            self.ebackup_azs[key_az_pnt] = list()
                        self.ebackup_azs[key_az_pnt].append(copy.deepcopy(ebk_s_info))
                        if not self.ebackup_azs.keys() or az not in self.ebackup_azs.keys():
                            self.ebackup_azs[az] = list()
                        self.ebackup_azs[az].append(copy.deepcopy(ebk_s_info))
            except Exception as msg:
                LOG.info("Read eBackup configuration failed. %s" % msg)
        else:
            cfg = config_parser.ConfigParser()
            cfg.read(ebackup_cfg_file)
            self.ebackup_server_ip = cfg.get("config", "ebackup_server_ip")
            self.ebackup_server_port = cfg.get("config", "ebackup_server_port")
            self.ebackup_login_username = cfg.get("config", "ebackup_login_username")
            self.ebackup_login_password = cfg.get("config", "ebackup_login_password_v2")
            self.ebackup_login_crt = cfg.get("config", "ebackup_login_crt")
            try:
                self.ebackup_servers = cfg.get("config", "ebackup_servers").split(',')
                for server in self.ebackup_servers:
                    ebackup_server_info = dict()
                    ebackup_server_info["ebackup_server_ip"] = cfg.get(server, 'ebackup_server_ip')
                    ebackup_server_info["ebackup_server_port"] = cfg.get(server, 'ebackup_server_port')
                    ebackup_server_info["ebackup_login_username"] = cfg.get(server, 'ebackup_login_username')
                    ebackup_server_info["ebackup_login_password"] = cfg.get(server, 'ebackup_login_password_v2')
                    ebackup_server_info["ebackup_login_crt"] = cfg.get(server, 'ebackup_login_crt')
                    azs = cfg.get(server, 'azs')
                    for az in azs:
                        self.ebackup_azs[az] = copy.deepcopy(ebackup_server_info)
            except Exception as msg:
                LOG.info("Read eBackup configuration failed. %s" % msg)

    def __get_conf_val(self, section, key):
        cmd = 'CINDER_CONF.{0}.{1}'.format(section, key)
        return eval(cmd)


class ErrorInfo(object):
    def __init__(self, error_code, error_desc="return error,but not has fail reason"):
        self.error_code = error_code
        self.error_desc = error_desc

    def __str__(self):
        error_json = {}
        error_json["code"] = str(self.error_code)
        error_json["desc"] = self.error_desc
        error_str = json.dumps(error_json)
        return error_str

    def set_error_info(self, error_code, error_desc):
        self.error_code = error_code
        self.error_desc = error_desc


def HandleErrorDesc(error_code):
    def func_wrapper(func):
        @functools.wraps(func)
        def real_wrapper(self, *argv, **kwargs):
            try:
                if func.__name__ in ["export_record", "import_record"]:
                    return func(self, *argv, **kwargs)
                else:
                    func(self, *argv, **kwargs)
            except Exception as msg:
                LOG.info(traceback.format_exc())
                LOG.error(str(msg))
                if not isinstance(argv[0], str):
                    # used in backup, restore, delete function
                    backupId = argv[0]['id']
                else:
                    backupId = argv[0]  # used in verify function
                if self.error_info.error_code == error_code or self.error_info.error_code == 0:
                    if "code" in str(msg):
                        json_msg = {}
                        try:
                            json_msg = json.loads(str(msg))
                        except Exception as error_msg:
                            LOG.info(error_msg)
                        if "code" in json_msg:
                            self._update_backup(backupId, fail_reason=str(msg))
                            raise exception.InvalidBackup(str(msg))
                        else:
                            self.error_info.set_error_info(error_code, str(msg))
                            self._update_backup(backupId, fail_reason=str(self.error_info))
                            raise exception.InvalidBackup(str(self.error_info))
                    else:
                        self.error_info.set_error_info(error_code, str(msg))
                        self._update_backup(backupId, fail_reason=str(self.error_info))
                        raise exception.InvalidBackup(str(self.error_info))
                else:
                    self._update_backup(backupId, fail_reason=str(self.error_info))
                    raise exception.InvalidBackup(str(self.error_info))

        return real_wrapper

    return func_wrapper


ebackup_conn_azs = None
is_infiniBand_env = False
connect_init_lock = threading.Lock()


class EbackupDriver(BackupDriverWithVerify):
    """Provides backup, restore and delete of volumes backup for eBackup."""

    DRIVER_VERSION = '1.0.0'
    __task_type = ''
    __backup = None
    __copy_image_volume_id = ''
    __copy_image_source_id = ''
    __driver_task_type = 0
    __driver_import_type = 0
    __is_image_has_copied = False
    __availability_zone = 'default'
    __fusionstorage_ip = ''
    __physical_network_type = ''
    __backup_orig_id = ''
    __version_info = ''
    __backup_info = []

    def __init__(self, context, db_driver=None):
        global ebackup_conn_azs
        global connect_init_lock
        connect_init_lock.acquire()
        if ebackup_conn_azs is None:
            LOG.info('Start init ebackup connect for ebackup Driver.')
            self.__create_connections()
            LOG.info('Init ebackup connect for ebackup Driver finished.')
        connect_init_lock.release()
        super(EbackupDriver, self).__init__(context, db_driver)
        self.backup_driver_name = 'cinder.backup.drivers.ebackup'
        self.ebackup_manager_unit_uri = '/v1/srv_jobmanager/get_single_brick'
        self.ebackup_manager_unit_list_uri = \
            '/v1/srv_jobmanager/get_list_of_bricks'
        self.ebackup_repository_uri = '/vbackup_repository'
        self.ebackup_manager_repository_uri = \
            '/v1/srv_jobmanager/get_single_namespace'
        self.ebackup_manager_repository_list_uri = \
            '/v1/srv_jobmanager/get_list_of_namespaces'
        self.ebackup_set_uri = '/vbackup_protectedset'
        self.ebackup_policy_uri = '/vbackup_policy'
        self.ebackup_plan_uri = '/vbackup_plan'
        self.ebackup_manager_plan_list_uri = \
            '/v1/srv_jobmanager/get_list_of_backup_plans'
        self.run_backup_plan_uri = '/v1/srv_jobmanager/run_backup_plan'
        self.ebackup_progress_uri = '/v1/srv_jobmanager/get_list_of_jobs'
        self.ebackup_task_uri = '/v1/srv_jobmanager'
        self.ebackup_image_uri = '/vbackup_image'
        self.ebackup_manager_single_snap = '/v1/srv_jobmanager/get_single_snap'
        self.ebackup_image_copy_uri = '/vbackup_image/copysnaps'
        self.ebackup_restore_disk_uri = '/vbackup_restore/disk'
        self.ebackup_get_backup_size = '/vbackup_backupobj'
        self.ebackup_copy_policy_uri = '/vcopy_policy'
        self.ebackup_copy_plan_uri = '/vcopy_plan'
        self.ebackup_restore_direct_uri = '/vbackup_restore_direct/disk'
        self.ebackup_restore_remote_uri = '/vbackup_restore_cross_region/disk'
        self.ebackup_encrypt_url = '/encrypt'
        self.not_do_lld_in_redo_restore = False
        self.error_info = ErrorInfo(0)
        self.evs_body = {}

    def init_evs_body(self, backup, typed):
        evs = EVS()
        if typed == 'backup':
            self.evs_body = evs.create_evs_body(self.db, self.context, backup, True)
        elif typed == 'restore' or typed == 'delete':
            self.evs_body = evs.decrypt_evs_body(self.db, self.context, backup, True)

    def __backup_get_by_volume_id(self, volume_id, get_pre_delete=False):
        '''db interface is different for M and J versioin'''
        backup_list = []
        if hasattr(self.db, 'backup_get_by_volume_id'):
            backup_list = call_db_fun(self.db.backup_get_by_volume_id, self.context, volume_id)
        elif hasattr(self.db, 'backup_get_all_by_volume'):
            backup_list = call_db_fun(self.db.backup_get_all_by_volume, self.context, volume_id)
        else:
            LOG.error("db interface for get backup from volume can not be found.")

        if get_pre_delete and hasattr(self.db, 'backup_get_all_with_deleted'):
            tmp_context = self.context.elevated()
            deleting_filter = {"status": "deleting", "volume_id": volume_id}
            backup_deleting = call_db_fun(self.db.backup_get_all_with_deleted, tmp_context, filters=deleting_filter)
            pre_delete_filter = {"status": "pre_deleted", "volume_id": volume_id}
            backup_pre_delete = call_db_fun(self.db.backup_get_all_with_deleted, tmp_context, filters=pre_delete_filter)
            backup_list.extend(backup_deleting)
            backup_list.extend(backup_pre_delete)
        return backup_list

    def __get_connection(self):
        conn_ebk = None
        try:
            global is_infiniBand_env
            if is_infiniBand_env:
                if self.__fusionstorage_ip:
                    elect_order = [(self.__availability_zone, self.__fusionstorage_ip),
                                   ('default', self.__fusionstorage_ip),
                                   (self.__availability_zone, 'default'),
                                   ('default', 'default')]
                    for target_server in elect_order:
                        if target_server in ebackup_conn_azs:
                            conn = ebackup_conn_azs[target_server]
                            break
                    if conn is None:
                        fail_msg = ("Failed to find the backup server based "
                                    "on target AZ {} and Huawei Distributed "
                                    "Block Storage Manager IP address {}."
                                    "".format(self.__availability_zone,
                                              self.__fusionstorage_ip))
                        LOG.error(fail_msg)
                        raise Exception(fail_msg)
                elif self.__physical_network_type:
                    elect_order = [(self.__availability_zone, self.__physical_network_type),
                                   ('default', self.__physical_network_type),
                                   (self.__availability_zone, ''),
                                   ('default', '')]
                    for target_server in elect_order:
                        if target_server in ebackup_conn_azs:
                            conn = ebackup_conn_azs[target_server]
                            break
                    if conn is None:
                        fail_msg = ("Failed to find the backup server based on target AZ %s and network type %s" % (
                            self.__availability_zone, self.__physical_network_type))
                        LOG.error(fail_msg)
                        raise Exception(fail_msg)
            else:
                conn = ebackup_conn_azs[self.__availability_zone]
            LOG.debug("find ebackup list for az: %s , FSM: %s , PN: %s" % (
                self.__availability_zone, self.__fusionstorage_ip, self.__physical_network_type))
            if self.__task_type in ['backup', 'copy']:
                exist_backup = self.__backup_get_by_volume_id(self.__backup['volume_id'], get_pre_delete=True)
                if len(exist_backup):
                    LOG.debug("start get ebackup UUID from nearest backup!")
                    history_uuid = ''
                    for bk in range(len(exist_backup)):
                        if exist_backup[bk].get('service_metadata'):
                            s_m = json.loads(exist_backup[bk].get('service_metadata'))
                            if s_m.get('e_u'):
                                history_uuid = s_m.get('e_u')
                                LOG.debug("got ebackup UUID %s from nearest backup!" % history_uuid)
                                break
                    if history_uuid:
                        for con_index in range(len(conn)):
                            if history_uuid and history_uuid == conn[con_index].ebackup_conf.uuid:
                                LOG.debug("found ebackup which UUID id %s!" % history_uuid)
                                conn_ebk = conn[con_index]
                                break
                    elif len(exist_backup) > 1:
                        for con_index in range(len(conn)):
                            if conn[con_index].ebackup_conf.extend == 'False':
                                LOG.info("found extend ebackup which UUID id %s!" % conn[con_index].ebackup_conf.uuid)
                                conn_ebk = conn[con_index]
                                break
                if conn_ebk is None:
                    LOG.info("Not found available ebackup from history backups, try to find it from driver's config.")
                    freedomest = -1
                    for con_index in range(len(conn)):
                        if conn[con_index].ebackup_conf.tenant_write_list:
                            if self.__backup.get('project_id') in conn[con_index].ebackup_conf.tenant_write_list:
                                conn_ebk = conn[con_index]
                                LOG.info('found eBackup specialed for tenant : %s, ip: %s' % (
                                    self.__backup['project_id'], conn_ebk.ebackup_conf.ebackup_server_ip))
                                break
                            else:
                                LOG.info('cant use eBackup with white-list: %s, current tenant: %s' % (
                                    conn[con_index].ebackup_conf.ebackup_server_ip, self.__backup['project_id']))
                                continue
                        try:
                            LOG.info('%s' % conn[con_index].ebackup_conf.ebackup_server_ip)
                            if conn[con_index] == conn[-1]:
                                c = EbackupConnectionClient(conn[con_index], True)
                            else:
                                c = EbackupConnectionClient(conn[con_index], False)
                            node_count, err_node, node_desc = c.get_backup_statistics()
                            bpc, err, desc = c.get_backup_plan_count()
                            cpc, err_c, desc_c = c.get_copy_plan_count()
                            if err == 0 and err_c == 0 and err_node == 0:
                                if int(node_count['FULLFUNCTIONALNODE']) == 0:
                                    node_count['FULLFUNCTIONALNODE'] = '1'
                                if ((int(bpc['COUNT']) + int(cpc['COUNT'])) / int(
                                                node_count['FULLFUNCTIONALNODE'])) < freedomest or freedomest == -1:
                                    conn_ebk = conn[con_index]
                                    freedomest = (int(bpc['COUNT']) + int(cpc['COUNT'])) / int(node_count['FULLFUNCTIONALNODE'])
                                    LOG.info("got freedomest conn %s %s [%s][%s][%s]!" % (
                                        conn[con_index].ebackup_conf.ebackup_server_ip,
                                        conn[con_index].ebackup_conf.uuid, bpc['COUNT'], cpc['COUNT'],
                                        node_count['FULLFUNCTIONALNODE']))
                        except Exception as msg:
                            LOG.error("got exception %s!" % msg)
                if conn_ebk:
                    _backup = call_db_fun(self.db.backup_get, self.context, self.__backup.get('id'))
                    json_service_metadata = _backup.get('service_metadata')
                    if json_service_metadata in (None, ''):
                        metadata = {}
                    else:
                        metadata = json.loads(json_service_metadata)
                    if metadata.get('e_u') != conn_ebk.ebackup_conf.uuid:
                        self._change_value_by_keys(metadata, e_u=conn_ebk.ebackup_conf.uuid)
                        self._update_backup(self.__backup['id'], service_metadata=json.dumps(metadata))
                        LOG.info('update backup backup_id : %s, uuid: %s' % (
                            self.__backup['id'], conn_ebk.ebackup_conf.uuid))
                else:
                    raise Exception("not found suitable ebackup!")
            if self.__task_type in ['restore', 'delete', 'export']:
                image_ebackup_uuid = ''
                if self.__backup.get('service_metadata'):
                    s_m = json.loads(self.__backup.get('service_metadata'))
                    if s_m.get('e_u'):
                        image_ebackup_uuid = s_m.get('e_u')
                for con_index in range(len(conn)):
                    if image_ebackup_uuid and image_ebackup_uuid == conn[con_index].ebackup_conf.uuid:
                        LOG.info("got ebackup from Image UUID %s!", image_ebackup_uuid)
                        conn_ebk = conn[con_index]
                        break
                    if not image_ebackup_uuid and conn[con_index].ebackup_conf.extend == 'False':
                        if not conn_ebk:
                            conn_ebk = conn[con_index]
                            LOG.info('get ebackup which not extend %s!', conn[con_index].ebackup_conf.uuid)
                        else:
                            conn_ebk = None
                            LOG.Error('error.more than one default ebackup!')
                            break
                # if not exist ,got first ebackup with volume az
                if not conn_ebk and len(conn) > 0 and self.__task_type == 'restore':
                    conn_ebk = conn[0]
        except Exception as msg:
            if 'default' in ebackup_conn_azs.keys():
                conn_ebk = ebackup_conn_azs['default'][0]
            if not conn_ebk or not conn_ebk.ebackup_conf.ebackup_server_ip:
                LOG.error(msg)
                fail_msg = ("Failed to find the backup server based on target AZ %s" % self.__availability_zone)
                LOG.error(fail_msg)
                raise Exception(fail_msg)
        return conn_ebk

    def __create_connections(self):
        global ebackup_conn_azs
        ebackup_conn_azs = dict()
        ebackup_server_conf = ebackup_server_config()
        ebackup_server_conf.init_conf()
        if ebackup_server_conf.ebackup_server_ip != "":
            ebackup_conn_azs['default'] = Connection(copy.deepcopy(ebackup_server_conf))
        LOG.info("keys %s" % list(ebackup_server_conf.ebackup_azs.keys()))
        for az_key, az_server_list in ebackup_server_conf.ebackup_azs.items():
            for az in az_server_list:
                LOG.info("az %s" % list(az.keys()))
                ebackup_conf = az
                ebackup_server_conf.ebackup_server_ip = ebackup_conf["ebackup_server_ip"]
                ebackup_server_conf.ebackup_server_port = ebackup_conf["ebackup_server_port"]
                ebackup_server_conf.ebackup_login_username = ebackup_conf["ebackup_login_username"]
                ebackup_server_conf.ebackup_login_password = ebackup_conf["ebackup_login_password"]
                ebackup_server_conf.ebackup_login_crt = ebackup_conf["ebackup_login_crt"]
                ebackup_server_conf.fusion_storage_ip = ebackup_conf["fusion_storage_ip"]
                ebackup_server_conf.physical_network_type = ebackup_conf["physical_network_type"]
                ebackup_server_conf.fusion_storage_agent_ip = ebackup_conf["fusion_storage_agent_ip"]
                ebackup_server_conf.uuid = ebackup_conf["uuid"]
                ebackup_server_conf.extend = ebackup_conf["extend"]
                if ebackup_conf["tenant_white_list"]:
                    ebackup_server_conf.tenant_write_list = ebackup_conf["tenant_white_list"].split(',')
                else:
                    ebackup_server_conf.tenant_write_list = list()
                global is_infiniBand_env
                if ebackup_conf["physical_network_type"] == 'ib':
                    LOG.info("is_infiniBand_env is set to True")
                    is_infiniBand_env = True
                if az_key not in ebackup_conn_azs.keys():
                    ebackup_conn_azs[az_key] = list()
                ebackup_conn_azs[az_key].append(Connection(copy.deepcopy(ebackup_server_conf)))

    def _get_task_detail(self, task_details, default_msg=""):
        try:
            error_info_json = json.loads(task_details)
            if error_info_json is None:
                return default_msg

            desc = ""
            if "code" not in error_info_json:
                return default_msg
            if "description" not in error_info_json:
                error_info_json["description"] = desc
            elif error_info_json["description"] is None:
                error_info_json["description"] = desc
            fail_detail = json.dumps(error_info_json)

            self.error_info.set_error_info(error_info_json['code'], error_info_json["description"])
            if "" == fail_detail:
                return default_msg
            return fail_detail
        except Exception as msg:
            LOG.error(msg)
            return default_msg

    def _convert_error(self, error_code, error_desc="eBackup return error"):
        error_json = {}
        error_json["code"] = str(error_code)
        error_json["desc"] = error_desc
        error_ret = json.dumps(error_json)
        return error_ret

    def get_task_info(self, task_id):
        ''' query task info '''
        task_url = self.ebackup_task_uri + "/get_single_jobs?id=%s" % task_id
        LOG.debug('get task info url:%s' % task_url)
        task_result = self.__get_connection().get(task_url)
        error_number = self._get_error(task_result)
        if error_number == ErrorCode.TSKMG_TASK_ID_NOT_EXIST:
            self.error_info.set_error_info(error_number, self._get_error_description(task_result))
            return error_number, 0, None
        if error_number != 0:
            self.error_info.set_error_info(error_number, self._get_error_description(task_result))
            msg = self._convert_error(error_number, self.error_info.error_desc)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        task_infos = self._get_data(task_result)
        return task_infos

    def _get_task_progress_and_details(self, meta_data, task_type=''):
        ''' query task progress '''
        progress = meta_data.get("PROGRESS")
        status = meta_data.get("STATUS")
        if status == cst.TASK_STATUS_STOP:
            # if status is stop , change it to failed
            status = cst.TASK_STATUS_FAILED

        if task_type != '':
            t_type = meta_data.get("TASKTYPE")
            msg = ('get task type is : %s.' % t_type)
            LOG.info(msg)
            if str(task_type) != str(t_type):
                LOG.error('task type mismatch, get task type is:%s, expected task type is:%s' % (
                    str(t_type), str(task_type)))
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)
        details = meta_data.get("TASKERRORINFO")
        return status, progress, details

    def _get_task_details_speed(self, meta_data):
        ret_status = True
        task_speed = -1

        task_speed_kb = meta_data.get("SPEED")
        if task_speed_kb in (None, ''):
            ret_status = False
            return ret_status, task_speed

        if -1 != int(task_speed_kb):
            MbToKb = cst.Mi / cst.Ki
            task_speed_mb = float(task_speed_kb) / MbToKb
            task_speed = round(task_speed_mb, 1)

        return ret_status, task_speed

    def _get_task_details_spacesavingrate(self, meta_data):
        task_space_savingrate = meta_data.get("SPACESAVINGRATE")
        return task_space_savingrate

    def _get_task_progress(self, task_id, task_type=''):
        task_infos = self.get_task_info(task_id)
        if not isinstance(task_infos, dict):
            return ErrorCode.TSKMG_TASK_ID_NOT_EXIST, None

        status, progress, details = self._get_task_progress_and_details(task_infos, task_type)
        return status, progress

    def _get_task_id(self, task_type, task_uuid, volume_uuid=''):
        '''
            get taskid by tasktype createtime
        '''
        if volume_uuid:
            task_filter = "?filter=TASKTYPE::" + str(task_type) + "%20and%20REQUEST_ID::" + str(
                task_uuid) + "%20and%20ASSOCIATEMACHINEID::" + volume_uuid + "&range=[0-15]" + '&sortby=CREATETIME,d'
        else:
            task_filter = "?filter=TASKTYPE::" + str(task_type) + "%20and%20REQUEST_ID::" + str(
                task_uuid) + "&range=[0-15]" + '&sortby=CREATETIME,d'
        task_url = self.ebackup_progress_uri + task_filter
        task_id_url = ('get task id url:%s' % task_url)
        LOG.info(task_id_url)
        count = 0
        while True:
            meta_result = self.__get_connection().get(task_url)
            count += 1
            meta_error = self._get_error(meta_result)
            if meta_error != 0:
                self.error_info.set_error_info(meta_error, self._get_error_description(meta_result))
                msg = convert_error(meta_error, self.error_info.error_desc)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)

            meta_data = self._get_data(meta_result)

            if len(meta_data) != 0:
                task_id = meta_data[0].get("ID")
                task_type_returned = meta_data[0].get("TASKTYPE")
                break
            time.sleep(cst.TIME_SLEEP_60)
            if count < cst.TASK_QUERY_COUNT:
                LOG.warn('try get task id again, get count:%s' % count)
                continue
            else:
                raise exception.InvalidBackup(cst.VERIFY_TASK_NOT_FOUND)

        if task_id is None:
            msg = 'get task id is None.'
            error_msg = convert_error(ErrorCode.TSKMG_TASK_ID_NOT_EXIST, msg)  # this error code need pararamter
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        if task_type != '':
            msg = ('task_type_returned is : %s.' % task_type_returned)
            LOG.info(msg)
            if str(task_type) != str(task_type_returned):
                LOG.error('task type mismatch, get task type is:%s, expected task type is:%s' % (
                    str(task_type_returned), str(task_type)))
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)
        return task_id

    def _get_error(self, response):
        ''' get error num from response '''
        dict_error = response['error']
        return dict_error['code']

    def _get_error_description(self, response):
        ''' get error description from response '''
        dict_error = response['error']
        return dict_error['description']

    def _get_data(self, response):
        ''' get data from response '''
        return response.get("data", {})

    def _get_object(self, uri, name, fuzzy=False):
        ''' get object by name
        '''
        if fuzzy == True:
            object_uri = uri + '?filter=NAME:' + name
        else:
            object_uri = uri + '?filter=NAME::' + name
        result = self.__get_connection().get(object_uri)
        error_code = self._get_error(result)
        if error_code != 0:
            self.error_info.set_error_info(error_code, self._get_error_description(result))
        return result

    def _get_object_by_filter(self, uri, filter):
        ''' get object by name
        '''
        object_uri = uri + filter
        result = self.__get_connection().get(object_uri)
        error_code = self._get_error(result)
        if error_code != 0:
            self.error_info.error_code = error_code
            self.error_info.error_desc = self._get_error_description(result)
        return result

    def _be_exist(self, uri, name):
        ''' find the object exist
        '''
        result = self._get_object(uri, name)
        error_code = self._get_error(result)
        if error_code != 0:
            return 0
        meta_data = self._get_data(result)
        if len(meta_data) != 0:
            return 1
        return 0

    def _get_copy_plan_by_name(self, uri, name):
        ''' get object by name
        '''
        plan = dict()
        result = self._get_object(uri, name)
        error_code = self._get_error(result)
        if error_code != 0:
            return plan
        meta_data = self._get_data(result)
        for p in meta_data:
            if p.get("NAME") == name:
                plan = p
        return plan

    def _get_filter_string(self, unit_types):
        filter_string = ""
        for unit_type in unit_types:
            if unit_type == UNIT_NAS:
                filter_string += "1|2|"
            elif unit_type == UNIT_SAN:
                filter_string += "6|"
            elif unit_type == UNIT_S3:
                filter_string += "7|"
        filter_string = filter_string[0:len(filter_string) - 1]
        if len(filter_string) == 1:
            filter_string = "filter=filesystemtype:" + filter_string
        else:
            filter_string = "filter=filesystemtype:[" + filter_string + "]"
        return filter_string

    def _get_storage_unit_info_base_handle(self, unit_types, level):
        filter_string = self._get_filter_string(unit_types)
        filters = "%20and%20runningstatus:[0|1]%20and%20level::{level}&" \
                  "range=[0-15]&sortby=FREECAPACITY,d".format(level=level)
        if level in [0, 1, 2]:
            get_v3_unit_id_url = self.ebackup_manager_unit_list_uri + "?" + \
                                 filter_string + filters
        else:
            msg = "unknown error!"
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        v3_result = self.__get_connection().get(get_v3_unit_id_url)
        v3_error = self._get_error(v3_result)
        if v3_error != 0:
            self.error_info.set_error_info(v3_error, self._get_error_description(v3_result))
            msg = self._convert_error(v3_error)
            LOG.error(msg)
            return None, None, None, None
        v3_data = self._get_data(v3_result)
        if len(v3_data) != 0:
            if int(v3_data[0].get("FREECAPACITY", 0)) <= 0:
                msg = self._convert_error(ErrorCode.BRICK_CAPACITY_NOT_ENOUGH, 'No enough space of the unit')
                LOG.error(msg)
                return None, None, None, None
            unit_len = len(v3_data)
            if unit_len == 1:
                v3_unit_path = v3_data[0].get("PATH")
                v3_unit_id = v3_data[0].get("ID")
                v3_unit_name = v3_data[0].get("NAME")
                v3_unit_filesystem_type = v3_data[0].get("FILESYSTEMTYPE")
                v3_unit_parent_name = v3_data[0].get("PARENTNAME")
            else:
                unit_max = int(v3_data[0].get("FREECAPACITY", 0)) / 1024
                unit_count_max = 1
                i = 1
                while i < unit_len:
                    if unit_max - int(v3_data[i].get("FREECAPACITY", 0)) / 1024 <= 300:
                        unit_count_max += 1
                        i += 1
                        continue
                    else:
                        break
                LOG.info('eBackup return unit_count_max, unit_count_max is:%d' % unit_count_max)
                if unit_count_max == 1:
                    v3_unit_path = v3_data[0].get("PATH")
                    v3_unit_id = v3_data[0].get("ID")
                    v3_unit_name = v3_data[0].get("NAME")
                    v3_unit_filesystem_type = v3_data[0].get("FILESYSTEMTYPE")
                    v3_unit_parent_name = v3_data[0].get("PARENTNAME")
                else:
                    v3_unit_path = v3_data[0].get("PATH")
                    v3_unit_id = v3_data[0].get("ID")
                    v3_unit_name = v3_data[0].get("NAME")
                    v3_unit_filesystem_type = v3_data[0].get("FILESYSTEMTYPE")
                    v3_unit_parent_name = v3_data[0].get("PARENTNAME")
                    get_v3_repository_url = \
                        self.ebackup_repository_uri + \
                        "/count?filter=PARENTNAME::" + v3_data[0].get("NAME")
                    v3_repository_result = self.__get_connection().get(get_v3_repository_url)
                    v3_repository_error = self._get_error(v3_repository_result)
                    if v3_repository_error != 0:
                        self.error_info.set_error_info(
                            v3_repository_error,
                            self._get_error_description(v3_repository_result))
                        msg = self._convert_error(v3_repository_error)
                        LOG.error(msg)
                        return None, None, None, None
                    v3_repository_data = self._get_data(v3_repository_result)
                    min_count = v3_repository_data.get("COUNT")
                    LOG.info('eBackup return min_count, min_count is:%s' % min_count)
                    j = 1
                    while j < unit_count_max:
                        get_v3_repository_url = \
                            self.ebackup_repository_uri + \
                            "/count?filter=PARENTNAME::" + \
                            v3_data[j].get("NAME")
                        v3_repository_result = self.__get_connection().get(get_v3_repository_url)
                        v3_repository_error = self._get_error(v3_repository_result)
                        if v3_repository_error != 0:
                            self.error_info.set_error_info(v3_repository_error,
                                                           self._get_error_description(v3_repository_result))
                            msg = self._convert_error(v3_repository_error)
                            LOG.error(msg)
                            return None, None, None, None
                        v3_repository_data = self._get_data(v3_repository_result)
                        count_temp = v3_repository_data.get("COUNT")
                        LOG.info('eBackup return count_temp, count_temp is:%s' % count_temp)
                        if int(count_temp) < int(min_count):
                            v3_unit_path = v3_data[j].get("PATH")
                            v3_unit_id = v3_data[j].get("ID")
                            v3_unit_name = v3_data[j].get("NAME")
                            v3_unit_filesystem_type = v3_data[j].get("FILESYSTEMTYPE")
                            v3_unit_parent_name = v3_data[j].get("PARENTNAME")
                            min_count = count_temp
                            j += 1
                        else:
                            j += 1
        else:
            LOG.error("get storage unit is failed, there is no storage unit")
            return None, None, None, None
        LOG.info("get storage unit is: %s" % v3_unit_path)
        return v3_unit_id, v3_unit_name, v3_unit_filesystem_type, v3_unit_parent_name

    def _get_storage_unit_info(self, unit_types):

        env_type = ebackup_driver_conf.ebackup_env_type
        unit_types_for_level_1 = list()
        unit_types_for_level_1.append(UNIT_S3)
        unit_types_for_level_1.append(UNIT_NAS)
        if env_type == ENV_HYBRID_CLOUD:
            unit_types_for_level_1.append(UNIT_SAN)
        level = 1
        v3_unit_id, v3_unit_name, v3_unit_filesystem_type, v3_unit_parent_name = self._get_storage_unit_info_base_handle(
            unit_types_for_level_1, level)
        if v3_unit_id is None and v3_unit_name is None and v3_unit_filesystem_type is None and v3_unit_parent_name is None:
            if len(unit_types) != 0:
                level = 0
                v3_unit_id, v3_unit_name, v3_unit_filesystem_type, v3_unit_parent_name = self._get_storage_unit_info_base_handle(
                    unit_types, level)

        return v3_unit_id, v3_unit_name, v3_unit_filesystem_type, v3_unit_parent_name

    def _create_store_unit(self, name, unit_path):
        en_uds_login_ak = ebackup_driver_conf.ebackup_storage_username
        en_uds_login_sk = ebackup_driver_conf.ebackup_storage_password
        uds_protocol = ebackup_driver_conf.ebackup_storage_s3_protocol
        uds_login_ak = Utils.decrypt_password(en_uds_login_ak)
        uds_login_sk = Utils.decrypt_password(en_uds_login_sk)

        ebk_client = EbackupConnectionClient(self.__get_connection())
        storage_unit, ret_code, error_msg = ebk_client.create_storage_unit(name,
                                                                           unit_path,
                                                                           uds_login_ak,
                                                                           uds_login_sk,
                                                                           uds_protocol)
        if ret_code != 0 or not storage_unit or "ID" not in storage_unit[0].keys():
            self.error_info.set_error_info(ret_code, error_msg)
            msg = self._convert_error(ret_code)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        return storage_unit[0].get("ID")

    def _get_public_cloud_storage_info(self, backup, is_cross_az_restore=False, special_s3_path=""):
        unit_filesystem_type = FILESYS_S3
        if not special_s3_path:
            tenant_id = self._get_tenant_id(backup, is_cross_az_restore)
            dec = self.get_dec_flag(backup)
            unit_name = self._get_unit_name(tenant_id, dec)
            pool_name = self._get_pool_name(tenant_id, dec)
            s3_unit_path = self._get_s3_unit_path(tenant_id, dec)
        else:
            s3_unit_path = special_s3_path
            slash = special_s3_path.rfind('/')
            if slash == -1:
                error_msg = 'invalid storage unit path'
                raise exception.InvalidBackup(error_msg)
            unit_name = special_s3_path[slash + 1:]
            pool_name = special_s3_path[slash + 1:]
        b_unit_exist = self._be_exist(
            self.ebackup_manager_unit_list_uri, unit_name)
        ebk_client = EbackupConnectionClient(self.__get_connection())
        if b_unit_exist == 0:
            LOG.info('dont exist storage unit, need to create it.')
            unit_id = self._create_store_unit(unit_name, s3_unit_path)
            ebk_client = EbackupConnectionClient(self.__get_connection())
            storage_pool, error_code, error_msg = ebk_client.create_storage_pool(pool_name, unit_id)
            if error_code != 0 or not storage_pool or "ID" not in storage_pool[0].keys():
                msg = 'storage pool id is None.'
                error_msg = convert_error(ErrorCode.POOL_NON_EXISTENT, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            pool_id = storage_pool[0]['ID']
        else:
            LOG.info('exist storage unit, get its id.')
            unit_id, pool_id = self._get_unit_info(
                self.ebackup_manager_unit_list_uri, unit_name, pool_name)
            if unit_id is None:
                msg = 'Storage unit id is None.'
                error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            if pool_id is None:
                msg = 'storage pool id is None.'
                error_msg = convert_error(ErrorCode.POOL_NON_EXISTENT, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            # get pool by id
            storage_pool, ret_code, msg = ebk_client.get_storage_pool_by_id(pool_id)
            if ret_code != 0:
                error_msg = convert_error(ret_code, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            pool_name = storage_pool.get('NAME', '')
        return unit_id, unit_filesystem_type, pool_id, pool_name

    def _get_private_cloud_storage_info(self, backup):
        backup_mode = self._get_copy_mode(backup)
        if (backup_mode == -1):
            msg = "get backup mode failed."
            LOG.info(msg)
            backup_mode = BACKUP_MODE_LOCAL
        unit_types = list()

        if backup_mode == BACKUP_MODE_TO_CLOUD:
            unit_types.append(UNIT_S3)
        else:
            if (ebackup_driver_conf.ebackup_cifs_data_layout != "") or (
                        ebackup_driver_conf.ebackup_nfs_data_layout != ""):
                unit_types.append(UNIT_NAS)
        if ebackup_driver_conf.ebackup_cifs_data_layout == "" and ebackup_driver_conf.ebackup_nfs_data_layout == "" and ebackup_driver_conf.ebackup_uds_data_layout == "":
            msg = 'Does not contain a valid configuration of datalayout.'
            error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        unit_id, unit_name, unit_filesystem_type, pool_name = self._get_storage_unit_info(unit_types)
        if unit_id is None:
            msg = 'No suitable storage unit was found.'
            error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        if pool_name == "":
            pool_name = unit_name

        ebk_client = EbackupConnectionClient(self.__get_connection())
        storage_pool, ret_code, error_msg = ebk_client.create_storage_pool(pool_name, unit_id)
        if ret_code != 0 or not storage_pool or "ID" not in storage_pool[0].keys():
            msg = 'Storage pool id is None.'
            error_msg = convert_error(ErrorCode.POOL_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        pool_id = storage_pool[0].get("ID")
        return unit_id, unit_filesystem_type, pool_id, pool_name

    def _get_hybrid_cloud_storage_info(self, backup):
        backup_mode = self._get_copy_mode(backup)
        if (backup_mode == -1):
            msg = "get backup mode failed."
            LOG.info(msg)
            return
        unit_types = []
        if backup_mode == BACKUP_MODE_TO_CLOUD:
            unit_types.append(UNIT_S3)
        else:
            if (ebackup_driver_conf.ebackup_cifs_data_layout != "") or (
                        ebackup_driver_conf.ebackup_nfs_data_layout != ""):
                unit_types.append(UNIT_NAS)
            if ebackup_driver_conf.ebackup_san_data_layout != "":
                unit_types.append(UNIT_SAN)
        if len(unit_types) == 0:
            msg = 'Get storage unit failed, please set the right backup storage.'
            error_msg = convert_error(ErrorCode.BD_STORAGE_NOT_SET, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        unit_id, unit_name, unit_filesystem_type, pool_name = self._get_storage_unit_info(unit_types)
        if unit_id is None:
            msg = 'No suitable storage unit was found'
            error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        if pool_name == "":
            pool_name = unit_name

        ebk_client = EbackupConnectionClient(self.__get_connection())
        storage_pool, error_code, error_msg = ebk_client.create_storage_pool(pool_name, unit_id)
        if error_code != 0 or not storage_pool or "ID" not in storage_pool[0].keys():
            msg = 'storage pool id is None.'
            error_msg = convert_error(ErrorCode.POOL_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        return unit_id, unit_filesystem_type, storage_pool[0]['ID'], pool_name

    def _get_storage_info(self, backup):
        env_type = ebackup_driver_conf.ebackup_env_type
        if env_type == ENV_PUBLIC_CLOUD:
            unit_id, unit_filesystem_type, pool_id, pool_name = self._get_public_cloud_storage_info(backup)
        elif env_type == ENV_PRIVATE_CLOUD:
            unit_id, unit_filesystem_type, pool_id, pool_name = self._get_private_cloud_storage_info(backup)
        elif env_type == ENV_HYBRID_CLOUD:
            unit_id, unit_filesystem_type, pool_id, pool_name = self._get_hybrid_cloud_storage_info(backup)
        else:
            LOG.error('wrong backup env type.')
            unit_id = None
            unit_filesystem_type = None
            pool_id = None
            pool_name = None
        return unit_id, unit_filesystem_type, pool_id, pool_name

    def _get_unit_info(self, uri, name, pool_name):
        '''
        _get_unit_info

        :param uri, name:
        :return:unitid, poolid
        '''
        result = self._get_object(uri, name)
        error_info = self._get_error(result)
        if error_info != 0:
            LOG.error('get object id failed, eBackup error num is: %d' % error_info)
            return None, None
        data_info = self._get_data(result)
        unitid = None
        poolid = None
        if len(data_info) != 0:
            unitid = data_info[0].get("ID")
            poolid = data_info[0].get("PARENTID")
        if poolid == '0':
            ebk_client = EbackupConnectionClient(self.__get_connection())
            result, error, error_msg = ebk_client.create_storage_pool(pool_name, unitid)
            if error != 0 or not result or "ID" not in result[0].keys():
                self.error_info.set_error_info(error, error_msg)
                msg = self._convert_error(error)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            poolid = result[0].get("ID")
        return unitid, poolid

    def _get_policy_name(self, layout_type):
        policy_name = "openstack_default_policy_nozip_noDedupe"
        if layout_type == '1':
            policy_name = "openstack_default_policy_nozip_Dedupe"
        elif layout_type == '2':
            policy_name = "openstack_default_policy_zip_noDedupe"
        elif layout_type == '3':
            policy_name = "openstack_default_policy_zip_Dedupe"
        elif layout_type == '4':
            policy_name = "openstack_default_policy_nozip_Distribute_Dedupe"
        elif layout_type == '6':
            policy_name = "openstack_default_policy_zip_Distribute_Dedupe"
        return policy_name

    def _get_copy_policy_name(self, layout_type):
        policy_name = "openstack_default_copy_policy_nozip_noDedupe"
        if layout_type == '2' or layout_type == '3':
            policy_name = "openstack_default_copy_policy_zip_noDedupe"
        if layout_type == '4' or layout_type == '6':
            policy_name = "openstack_default_copy_policy_zip_distirbuteDedupe"
        return policy_name

    def _get_data_layout_by_pool(self, pool_id):
        ebk_client = EbackupConnectionClient(self.__get_connection())
        storage_pool, error_code, error_mgs = ebk_client.get_storage_pool_by_id(str(pool_id))

        if error_code != 0:
            msg = self._convert_error(error_code, error_mgs)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        unit_list = storage_pool["BRICKLIST"]
        if len(unit_list) == 0:
            msg = 'unit is null.'
            error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        unit_list_py = json.loads(unit_list)
        unit_info = unit_list_py[0]
        unit_filter = "?id=" + str(unit_info["ID"])

        result = self._get_object_by_filter(
            self.ebackup_manager_unit_uri, unit_filter)
        error_code = self._get_error(result)
        if error_code != 0:
            msg = self._convert_error(error_code, "get storage unit failed")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        unit_meta_data = self._get_data(result)

        if unit_meta_data["FILESYSTEMTYPE"] == "1":  # NFS
            return ebackup_driver_conf.ebackup_cifs_data_layout
        elif unit_meta_data["FILESYSTEMTYPE"] == "2":
            return ebackup_driver_conf.ebackup_nfs_data_layout
        elif unit_meta_data["FILESYSTEMTYPE"] == "7":
            return ebackup_driver_conf.ebackup_uds_data_layout
        else:
            return None

    def _get_data_layout_from_name(self, policy_name):
        if policy_name == "openstack_default_policy_nozip_noDedupe":
            return 0
        elif policy_name == "openstack_default_policy_nozip_Dedupe":
            return 1
        elif policy_name == "openstack_default_policy_zip_noDedupe" or policy_name == "openstack_default_policy":
            return 2
        elif policy_name == "openstack_default_policy_zip_Dedupe":
            return 3
        elif policy_name == "openstack_default_policy_nozip_Distribute_Dedupe":
            return 4
        elif policy_name == "openstack_default_policy_zip_Distribute_Dedupe":
            return 6
        else:
            return None

    def _get_tenant_id(self, backup, is_cross_az_restore=False):
        if is_cross_az_restore:
            json_service_metadata = backup['service_metadata']
            if json_service_metadata in (None, ''):
                LOG.info('service metadata is null, use project_id in backup to restore')
                tenant_id = backup["project_id"]
            else:
                meta_data = json.loads(json_service_metadata)
                try:
                    BP = meta_data.get("BP")
                    if len(BP) > 32:
                        uuid.UUID(BP[-32:])  # verify it is a uuid
                        tenant_id = BP[-32:]
                    else:
                        tenant_id = backup["project_id"]
                except Exception as msg:
                    LOG.info(msg)
                    tenant_id = backup["project_id"]
        else:
            tenant_id = backup["project_id"]
        return tenant_id

    def _get_cascaded_volume_id(self, backup, ana_metadata=True):
        volume_id = backup['volume_id']
        if ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD:
            try:
                source_vol = call_db_fun(self.db.volume_get, self.context, volume_id)
                if source_vol and source_vol.get('vendor_driver_metadata', ''):
                    try:
                        driver_m = json.loads(source_vol.get('vendor_driver_metadata'))
                        if driver_m.get('alias_volume_id'):
                            volume_id = driver_m.get('alias_volume_id')
                            LOG.info('get original volume id %s by alias_volume_id' % volume_id)
                    except Exception as e:
                        LOG.error('load json of vendor_driver_metadata failed %s!' % e)
                elif backup.get('service_metadata', '') and ana_metadata:
                    try:
                        md = json.loads(backup.get('service_metadata', ''))
                        if md.get('SP'):
                            volume_id = md.get('SP').split('/')[-1]
                            LOG.info('get original volume id %s by metadata SP' % volume_id)
                    except Exception as e:
                        LOG.error('load json of service_metadata failed %s!' % e)
            except Exception as e:
                LOG.error('get volume_id failed %s!' % e)
        flag = self._check_length(volume_id, cst.LENGTH_LIMIT_36)
        if flag:
            msg = ('volume id length does not comply, error length:%s' % len(volume_id))
            error_msg = convert_error(ErrorCode.BD_VOLUME_ID_INVALID, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        return volume_id

    def _get_rep_name(self, backup):
        tenant_id = self._get_tenant_id(backup)
        env_type = ebackup_driver_conf.ebackup_env_type
        dec = self.get_dec_flag(backup)
        if dec:
            repository_name = 'openstack_repository_dec_%s' % tenant_id
        else:
            repository_name = 'openstack_repository_%s' % tenant_id
        if env_type == ENV_HYBRID_CLOUD or env_type == ENV_PRIVATE_CLOUD:
            backup_mode = self._get_copy_mode(backup)
            if backup_mode == -1 and env_type == ENV_PRIVATE_CLOUD:
                backup_mode = BACKUP_MODE_LOCAL
            if backup_mode == -1:
                msg = 'get backup mode failed.'
                error_msg = convert_error(ErrorCode.BD_GET_BACKUP_MODE_FAILED, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            if backup_mode == BACKUP_MODE_LOCAL and env_type == ENV_PRIVATE_CLOUD:
                return repository_name
            repository_name = repository_name + '_' + str(backup_mode)
        return repository_name

    def _get_copy_target_rep_name(self, backup):
        tenant_id = self._get_tenant_id(backup)
        dec = self.get_dec_flag(backup)
        backup_metadata = Utils.transfer_json_to_dict(backup['service_metadata'])
        if backup_metadata.get('IT') == cst.IMPORT_TYPE_COPY and dec:
            repository_name = 'openstack_repository_target_dec_%s' % tenant_id
        elif backup_metadata.get('IT') == cst.IMPORT_TYPE_REMOTE_REPLICATION and dec:
            repository_name = 'openstack_repository_replication_dec_%s' % tenant_id
        elif backup_metadata.get('IT') == cst.IMPORT_TYPE_COPY:
            repository_name = 'openstack_repository_target_%s' % tenant_id
        elif backup_metadata.get('IT') == cst.IMPORT_TYPE_REMOTE_REPLICATION:
            repository_name = 'openstack_repository_replication_%s' % tenant_id
        else:
            msg = ('backup[%s] import type is wrong:%s' % (backup['id'], backup_metadata.get('IT')))
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        return repository_name

    def _get_plan_name_without_datalayout(self, policy_name_with_datalayout):
        plan_list = policy_name_with_datalayout.split('_')
        plan_name = ""
        count = len(plan_list)
        for i in range(0, count - 2):
            plan_name += plan_list[i]
            plan_name += '_'
        plan_name = plan_name[:-1]
        LOG.info("plan_name_without_datalayout: %s", plan_name)
        return plan_name

    def check_lld_enable(self, backup):
        if ebackup_driver_conf.ebackup_lazyloading == '0':
            LOG.info("lld is disable or forbidden")
            return False

        LOG.info("context.roles: %s", self.context.roles)
        inc = self._get_backup_mode_from_inc(backup)
        if ebackup_driver_conf.ebackup_lazyloading in ['1', '3'] and (
                        'op_gated_bklld' in self.context.roles or inc != cst.BACKUP_CMD_FULL):
            LOG.info("lld is disabled in full mode")
            return False
        elif ebackup_driver_conf.ebackup_lazyloading in ['2', '4'] and inc != cst.BACKUP_CMD_FULL:
            LOG.info("Dose not check the lld test flag, and lld is disabled in full mode")
            return False

        if backup.get(cst.LLD_DISABLED, None) is not None:
            LOG.info("lld is disable in backup")
            return False
        return True

    def _get_plan_name(self, backup, extra_info=None):
        volume_id = self._get_cascaded_volume_id(backup)
        backup_id = backup['id']
        env_type = ebackup_driver_conf.ebackup_env_type
        backup_result = call_db_fun(self.db.backup_get, self.context, backup_id)
        json_metadata = backup_result.get('service_metadata')
        metadata = {}
        if json_metadata not in ("", None):
            metadata = json.loads(json_metadata)
        # found type of backup storage which we should get
        if ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD:
            # public cloud support S3 only
            unit_filesystem_type = FILESYS_S3
        elif ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD or ebackup_driver_conf.ebackup_env_type == ENV_HYBRID_CLOUD:
            # private we need get it from exist pool first.
            # we should protect all of the volume's plan belong the same repository.
            LOG.info("search storage type from exist repository first")
            repository = ""
            vbs_plan_short = "openstack_plan_" + volume_id
            plan = self._get_object(
                self.ebackup_manager_plan_list_uri, vbs_plan_short, fuzzy=True)
            error_code = self._get_error(plan)
            if error_code != 0:
                msg = self._convert_error(error_code, "get backup plan failed")
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            vbsplandata = self._get_data(plan)
            if len(vbsplandata) != 0:
                repository = vbsplandata[0].get('NAMESPACENAME')
            else:
                csbs_plan_short = "openstack_csbs_plan_" + volume_id
                csbs_plan = self._get_object(
                    self.ebackup_manager_plan_list_uri, csbs_plan_short,
                    fuzzy=True)
                error_code = self._get_error(csbs_plan)
                if error_code != 0:
                    msg = self._convert_error(error_code, "get backup plan failed")
                    LOG.error(msg)
                    raise exception.InvalidBackup(msg)
                csbsplandata = self._get_data(csbs_plan)
                if len(csbsplandata) != 0:
                    LOG.info('plan: %s' % csbsplandata[0])
                    repository = csbsplandata[0].get('NAMESPACENAME')
            if repository != "":
                LOG.info("get repository named:%s" % repository)
                _N_, unit_filesystem_type, _N_, _N_ = self.get_more_info_by_rep_name(repository)
            else:
                # we will choose the biggest storage unit second
                LOG.info("search storage type from biggest unit second")
                _N_, unit_filesystem_type, _N_, _N_ = self._get_private_cloud_storage_info(backup)
            LOG.info("get storage unit type is:%s" % unit_filesystem_type)
        else:
            msg = 'unknown cloud type!'
            self.error_info.set_error_info(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        policy_name = self._get_policy_name_by_filesys_type(unit_filesystem_type)
        if extra_info is not None and isinstance(extra_info, dict):
            extra_info['policy_name'] = policy_name
        short_policy_name = policy_name.split('openstack_default_policy_')[1].lower()
        st = metadata.get('ST', 0)
        dec = self.get_dec_flag(backup)
        BackupPlan = collections.namedtuple('backup_plan', ('st', 'dec'))
        plan_dict = {BackupPlan(0, 0): 'openstack_plan_%s_%s' % (volume_id, short_policy_name),
                     BackupPlan(1, 0): 'openstack_csbs_plan_%s_%s' % (volume_id, short_policy_name),
                     BackupPlan(0, 1): 'openstack_plan_dec_%s_%s' % (volume_id, short_policy_name),
                     BackupPlan(1, 1): 'openstack_csbs_plan_dec_%s_%s' % (volume_id, short_policy_name)
                     }
        plan_key = BackupPlan(st, dec)
        plan_name = plan_dict[plan_key]

        if env_type == ENV_HYBRID_CLOUD or env_type == ENV_PRIVATE_CLOUD:
            backup_mode = self._get_copy_mode(backup)
            if backup_mode == -1 and env_type == ENV_PRIVATE_CLOUD:
                backup_mode = BACKUP_MODE_LOCAL
            if backup_mode == -1:
                msg = 'get backup mode failed.'
                error_msg = convert_error(ErrorCode.BD_GET_BACKUP_MODE_FAILED, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            if backup_mode == BACKUP_MODE_LOCAL and env_type == ENV_PRIVATE_CLOUD:
                return plan_name
            plan_name = plan_name + '_' + str(backup_mode)

        return plan_name

    def get_remote_replication_plan_name(self, backup, datalayout=None):
        metadata = Utils.transfer_json_to_dict(backup.get('service_metadata'))
        if datalayout is None:
            datalayout = metadata.get('DL')
        suffix = 'nozip_nodedupe' if datalayout < 2 else 'zip_nodedupe'
        ReplicationPlan = collections.namedtuple('replication_plan', ('cbr', 'st', 'dec'))
        plan_name = {ReplicationPlan(0, 4, 0): 'openstack_rep_plan_%s_%s' % (backup['volume_id'], suffix),
                     ReplicationPlan(0, 5, 0): 'openstack_csbs_rep_plan_%s_%s' % (backup['volume_id'], suffix),
                     ReplicationPlan(0, 4, 1): 'openstack_rep_plan_dec_%s_%s' % (backup['volume_id'], suffix),
                     ReplicationPlan(0, 5, 1): 'openstack_csbs_rep_plan_dec_%s_%s' % (backup['volume_id'], suffix),
                     ReplicationPlan(1, 4, 0): 'openstack_cbr_rep_volume_plan_%s_%s' % (backup['volume_id'], suffix),
                     ReplicationPlan(1, 5, 0): 'openstack_cbr_rep_machine_plan_%s_%s' % (backup['volume_id'], suffix)
                     }
        replication_plan_name_key = ReplicationPlan(metadata.get('CBR', 0), metadata.get('ST'),
                                                    self.get_dec_flag(backup))
        rep_plan_name = plan_name[replication_plan_name_key]
        return rep_plan_name

    def _get_copy_plan_name(self, backup):
        tenant_id = self._get_tenant_id(backup)
        metadata = json.loads(backup.get('service_metadata'))
        if cst.COPY_TYPE_SINGLE_VOL == metadata.get('ST', 0):
            copy_plan_name = 'openstack_copy_plan_%s' % tenant_id
        else:
            copy_plan_name = 'openstack_csbs_copy_plan_%s' % tenant_id
        return copy_plan_name

    def _get_unit_name(self, tenant_id, dec):
        if dec:
            unit_name = 'openstack_unit_dec_%s' % tenant_id
        else:
            unit_name = 'openstack_unit_%s' % tenant_id
        return unit_name

    def _get_pool_name(self, tenant_id, dec):
        if dec:
            pool_name = 'openstack_pool_dec_%s' % tenant_id
        else:
            pool_name = 'openstack_pool_%s' % tenant_id
        return pool_name

    def _get_s3_unit_path(self, tenant_id, dec):
        if dec:
            s3_unit_path = ebackup_driver_conf.ebackup_storage_unit_path + ":/" + "csbs-dec-" + str(tenant_id)
        else:
            s3_unit_path = ebackup_driver_conf.ebackup_storage_unit_path + ":/" + \
                           ebackup_driver_conf.ebackup_bucket_name + str(tenant_id)
        return s3_unit_path

    def _get_policy_name_by_filesys_type(self, unit_filesystem_type):
        policy_name = "openstack_default_policy_nozip_noDedupe"
        if unit_filesystem_type == FILESYS_CIFS:
            policy_name = self._get_policy_name(ebackup_driver_conf.ebackup_cifs_data_layout)
        elif unit_filesystem_type == FILESYS_NFS:
            policy_name = self._get_policy_name(ebackup_driver_conf.ebackup_nfs_data_layout)
        elif unit_filesystem_type == FILESYS_S3:
            policy_name = self._get_policy_name(ebackup_driver_conf.ebackup_uds_data_layout)
        elif unit_filesystem_type == FILESYS_SAN:
            policy_name = self._get_policy_name(ebackup_driver_conf.ebackup_san_data_layout)
        return policy_name

    def _get_copy_policy_name_by_filesys_type(self, unit_filesystem_type):
        policy_name = "openstack_default_copy_policy_nozip_noDedupe"
        if unit_filesystem_type == FILESYS_CIFS:
            policy_name = self._get_copy_policy_name(ebackup_driver_conf.ebackup_cifs_data_layout)
        elif unit_filesystem_type == FILESYS_NFS:
            policy_name = self._get_copy_policy_name(ebackup_driver_conf.ebackup_nfs_data_layout)
        elif unit_filesystem_type == FILESYS_S3:
            policy_name = self._get_copy_policy_name(ebackup_driver_conf.ebackup_uds_data_layout)
        elif unit_filesystem_type == FILESYS_SAN:
            policy_name = self._get_copy_policy_name(ebackup_driver_conf.ebackup_san_data_layout)
        return policy_name

    def _check_unit_free_capacity(self, unit_id):
        select_count = 0
        while True:
            object_uri = self.ebackup_manager_unit_uri + '?id=' + unit_id
            result = self.__get_connection().get(object_uri)
            meta_data = self._get_data(result)
            fre_cap = meta_data.get('FREECAPACITY')
            if fre_cap not in ('0', None):
                return fre_cap
            select_count += 1
            if select_count > cst.QUERY_COUNT:
                msg = 'storage unit capacity is wrong!'
                self.error_info.set_error_info(ErrorCode.BD_GET_STORAGE_UNIT_INFO_FAILED, msg)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            time.sleep(cst.TIME_SLEEP_5 * 2)

    def _create_copy_plan(self, copy_plan_name, copy_policy_name, rep_src_name, rep_target_name):
        rep_body = {
            "NAME": copy_plan_name,
            "DESCRIPTION": "",
            "POLICYNAME": copy_policy_name,
            "STATUS": 0,
            "SROUCEREPOSITORYNAME": rep_src_name,
            "TARGETREPOSITORYNAME": rep_target_name
        }
        rep_body_str = 'create copy plan body is:' + str(rep_body)
        LOG.info(rep_body_str)
        result = self.__get_connection().post(self.ebackup_copy_plan_uri, rep_body)
        error_number = self._get_error(result)
        self.error_info.set_error_info(error_number, self._get_error_description(result))
        return error_number

    def _select_storage_unit(self, backup, unit_types, level, specific_unit=None):
        repository_target = self._get_copy_target_rep_name(backup)
        if specific_unit is None:
            unit_id, unit_name, unit_filesystem_type, target_pool_name = self._get_storage_unit_info_base_handle(
                unit_types,
                level)
            if unit_id is None:
                msg = 'No suitable storage unit was found'
                error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
                LOG.error(error_msg)
                return None, None
        else:
            unit_name = specific_unit['unit_name']
            unit_id = specific_unit['unit_id']
            unit_filesystem_type = specific_unit['unit_filesystem_type']
            target_pool_name = specific_unit['target_pool_name']

        if target_pool_name == "":
            target_pool_name = unit_name

        ebk_client = EbackupConnectionClient(self.__get_connection())
        storage_pool, error_code, error_msg = ebk_client.create_storage_pool(target_pool_name, unit_id)
        if error_code != 0 or not storage_pool or "ID" not in storage_pool[0].keys():
            msg = 'storage pool id is None.'
            error_msg = convert_error(ErrorCode.POOL_NON_EXISTENT, msg)
            LOG.error(error_msg)
            return None, None
        pool_id = storage_pool[0].get("ID")
        b_rep_result = self._get_object(
            self.ebackup_manager_repository_list_uri, repository_target)
        error_code = self._get_error(b_rep_result)
        if error_code != 0:
            error_msg = convert_error(error_code, "get target storage repository fail")
            LOG.error(error_msg)
            return None, None
        meta_data = self._get_data(b_rep_result)
        if len(meta_data) == 0:
            if ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD and repository_target.find("target") > 0:
                repository_target = repository_target + "_" + target_pool_name
            LOG.info('the repository dose not exist, need to create it with name %s.' % repository_target)
            fre_cap = self._check_unit_free_capacity(unit_id)
            ebkconn = EbackupConnectionClient(self.__get_connection())
            data, err, desc = ebkconn.create_storage_repository(repository_target, pool_id, target_pool_name, fre_cap)
            if err:
                msg = self._convert_error(err, "create target storage repository fail")
                LOG.error(msg)
                return None, None
        else:
            for target_repository in meta_data:
                if target_repository["NAME"] == repository_target + "_" + target_pool_name:
                    repository_target = target_repository["NAME"]
                    LOG.info('found accurate repository with poll name %s.' % repository_target)
        return unit_filesystem_type, repository_target

    def _create_copy_related_resource(self, backup, repository_src):
        env_type = ebackup_driver_conf.ebackup_env_type
        if not repository_src:
            copy_plan_name = self.get_remote_replication_plan_name(backup)
        else:
            copy_plan_name = self._get_copy_plan_name(backup)
            # check copy plan with volume id first
            copy_plan_name_with_volume = copy_plan_name + "_" + backup["volume_id"]
            plan_with_volume = self._get_copy_plan_by_name(
                self.ebackup_copy_plan_uri, copy_plan_name_with_volume)
            if plan_with_volume:
                LOG.info('Exist copy plan with volumeid, no need to create it.')
                return copy_plan_name_with_volume
        plan_without_volume = self._get_copy_plan_by_name(
            self.ebackup_copy_plan_uri, copy_plan_name)
        if plan_without_volume:
            LOG.info('Exist copy plan, no need to create it.')
            return copy_plan_name
        if env_type == ENV_PRIVATE_CLOUD:
            unit_types = [UNIT_S3, UNIT_NAS]
        elif env_type == ENV_HYBRID_CLOUD:
            unit_types = [UNIT_S3, UNIT_NAS, UNIT_SAN]
        elif env_type == ENV_PUBLIC_CLOUD:
            unit_types = [UNIT_S3]
        else:
            msg = "The env_type is error"
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        if env_type != ENV_PUBLIC_CLOUD:
            level = 2
            if not repository_src:
                level = 1
            unit_filesystem_type, repository_target = self._select_storage_unit(backup, unit_types, level)
            if unit_filesystem_type is None:
                unit_types = [UNIT_S3]
                level = 0
                unit_filesystem_type, repository_target = self._select_storage_unit(backup, unit_types, level)
            if unit_filesystem_type is None:
                msg = "no storage unit"
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
        else:
            level = 0
            unit_id, unit_filesystem_type, pool_id, pool_name = self._get_public_cloud_storage_info(backup)
            if unit_id is None:
                msg = 'Storage unit id is None.'
                error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            unit_info = {'unit_id': unit_id,
                         'unit_name': self._get_unit_name(backup['project_id'], self.get_dec_flag(backup)),
                         'unit_filesystem_type': unit_filesystem_type,
                         'target_pool_name': pool_name}
            unit_filesystem_type, repository_target = self._select_storage_unit(backup, unit_types, level, unit_info)

        if repository_src:
            copy_policy_name = self._get_copy_policy_name_by_filesys_type(unit_filesystem_type)
        else:
            copy_policy_name = 'openstack_default_copy_policy_nozip_noDedupe' if copy_plan_name.find('nozip') > 0 else \
                'openstack_default_copy_policy_zip_noDedupe'
        if repository_src:
            copy_plan_name = copy_plan_name + "_" + self._get_cascaded_volume_id(backup, False)
        ret = self._create_copy_plan(copy_plan_name, copy_policy_name, repository_src, repository_target)
        if ret == ErrorCode.EXISTS_NAME:
            LOG.info('copy plan already exists.')
            return copy_plan_name
        elif ret != 0:
            msg = self._convert_error(ret, "create copy plan fail")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        else:
            return copy_plan_name

    def _update_hybrid_related_resource(self, backup, snap_repository_name):
        ''' process of update backup plan '''
        copy_plan_name = self._get_copy_plan_name(backup)
        result = self._get_object(self.ebackup_copy_plan_uri, copy_plan_name)
        error_code = self._get_error(result)
        if error_code != 0:
            return
        meta_data = self._get_data(result)
        if len(meta_data) == 0:
            LOG.info('No Exist copy plan, no need to update it.')
            return
        source_respository_name = meta_data[0].get("SROUCEREPOSITORYNAME")
        if source_respository_name != "":
            LOG.info('source repository name exist, no need to update.')
            return
        update_copy_uri = self.ebackup_copy_plan_uri + '/' + meta_data[0].get("ID")
        update_copy_body = {
            "SROUCEREPOSITORYNAME": snap_repository_name
        }
        request_str = 'update copy plan body is:' + str(update_copy_body)
        LOG.info(request_str)
        result = self.__get_connection().put(update_copy_uri, update_copy_body)
        error_number = self._get_error(result)
        if error_number != 0:
            self.error_info.set_error_info(error_number, self._get_error_description(result))
            msg = self._convert_error(error_number, "update copy plan fail")
            LOG.error(msg)
            raise Exception(msg)
        return

    def _update_backup_plan_by_policy(self, plan_name, extra_info):
        LOG.info("update_backup_plan: %s", plan_name)
        plan_name_without_datalayout = self._get_plan_name_without_datalayout(plan_name)
        policy_exist = False
        result = self._get_object(
            self.ebackup_manager_plan_list_uri, plan_name_without_datalayout)
        error_code = self._get_error(result)
        if error_code != 0:
            return False
        backup_plan_list = self._get_data(result)
        if len(backup_plan_list) <= 0:
            return False

        policy_name = extra_info['policy_name']

        for plan in backup_plan_list:
            LOG.info("exist plan plicy name and new policy name: %s, %s", plan['POLICYNAME'], policy_name)
            LOG.info("exist plan: %s", plan)
            if plan['POLICYNAME'].lower() == policy_name.lower():
                policy_exist = True
                backup_plan_uri = self.ebackup_plan_uri + '/' + plan['ID']
                body = {"NAME": plan_name}
                LOG.info("backup_plan_uri:%s", backup_plan_uri)
                change_result = self.__get_connection().put(backup_plan_uri, body)
                resp_error = change_result['error']
                resp_error_code = resp_error['code']
                if resp_error_code != 0:
                    self.error_info.set_error_info(resp_error_code, self._get_error_description(change_result))
                    LOG.info("resp_error_code:%s", resp_error_code)
                    return False
        LOG.info('update plan has the same data layout? %s', policy_exist)
        return policy_exist

    def _get_copy_plan_repository(self, backup):
        tenant_id = self._get_tenant_id(backup)
        vbs_copy_plan = 'openstack_copy_plan_%s' % tenant_id
        vbs_copy_plan_with_volumeid = vbs_copy_plan + '_' + self._get_cascaded_volume_id(backup, False)
        csbs_copy_plan = 'openstack_csbs_copy_plan_%s' % tenant_id
        csbs_copy_plan_with_volumeid = csbs_copy_plan + '_' + self._get_cascaded_volume_id(backup, False)
        copy_plan_sequence = [vbs_copy_plan_with_volumeid, csbs_copy_plan_with_volumeid, vbs_copy_plan, csbs_copy_plan]
        for cp_cplan_item in copy_plan_sequence:
            plan_object = self._get_object(self.ebackup_copy_plan_uri, cp_cplan_item)
            error_code = self._get_error(plan_object)
            if error_code != 0:
                msg = self._convert_error(error_code, "get copy plan failed")
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            plan = self._get_data(plan_object)
            if len(plan) != 0:
                LOG.info("found plan %s" % cp_cplan_item)
                return plan[0].get('SROUCEREPOSITORYNAME')
        return ''

    def _check_storage_ralate(self, backup):
        ''' process of create backup plan '''
        source_volume_id = self._get_cascaded_volume_id(backup)
        prepare_start = (source_volume_id + ' Begin prepare backup.')
        LOG.info(prepare_start)
        policy_name = "openstack_default_policy_nozip_Dedupe"
        set_name = 'openstack_default_set'
        extra_info = {}
        plan_name = self._get_plan_name(backup, extra_info)
        repository_name = self._get_rep_name(backup)
        backup[cst.PLAN_NAME] = plan_name
        object_name = {"set_name": set_name,
                       "repository_name": repository_name,
                       "policy_name": policy_name,
                       "plan_name": plan_name
                       }

        plan_name_without_data_layout = self._get_plan_name_without_datalayout(plan_name)
        b_plan_exist = self._be_exist(
            self.ebackup_manager_plan_list_uri, plan_name)
        if b_plan_exist != 0:
            LOG.info('Exist plan, no need to create it.')
            return

        b_plan_exist = self._be_exist(
            self.ebackup_manager_plan_list_uri, plan_name_without_data_layout)
        if b_plan_exist != 0:
            if ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD and self.check_lld_enable(backup):
                update_result = self._update_backup_plan_by_policy(plan_name, extra_info)
                if update_result:
                    return
            elif ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD and not self.check_lld_enable(backup):
                LOG.info("old plan[%s] exist and lazyloading is disabled", plan_name_without_data_layout)
                backup[cst.PLAN_NAME] = plan_name_without_data_layout
                return
            else:
                update_result = self._update_backup_plan_by_policy(plan_name, extra_info)
            if update_result:
                return
        # check the backup plan was created during upgrade lld but not switch datalayout
        if ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD:
            plan_name_with_zip_nodedupe = plan_name_without_data_layout + '_zip_nodedupe'
            b_plan_exist = self._be_exist(
                self.ebackup_manager_plan_list_uri,
                plan_name_with_zip_nodedupe)
            if b_plan_exist != 0:
                if not self.check_lld_enable(backup):
                    LOG.info("user old plan[%s]", plan_name_with_zip_nodedupe)
                    backup[cst.PLAN_NAME] = plan_name_with_zip_nodedupe
                    return

        result = self._get_object(
            self.ebackup_manager_repository_list_uri, repository_name, True)
        error_code = self._get_error(result)
        if error_code != 0:
            msg = self._convert_error(error_code, "get repository failed")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        rep_meta_data = self._get_data(result)
        b_rep_exist = len(rep_meta_data)
        if b_rep_exist != 0:
            LOG.info('no need to create repository again, found repostory count:%d!' % len(rep_meta_data))
            exist_plan_list_obj = self._get_object(
                self.ebackup_manager_plan_list_uri,
                plan_name_without_data_layout, True)
            exist_plan_list = self._get_data(exist_plan_list_obj)
            copyplan_source_repository = self._get_copy_plan_repository(backup)
            rep_data = {}
            if len(exist_plan_list) > 0:
                for repository in rep_meta_data:
                    if repository.get("NAME") == exist_plan_list[0].get("NAMESPACENAME"):
                        rep_data = repository
                if not rep_data:
                    msg = 'Storage repository Not Found.'
                    error_msg = convert_error(ErrorCode.REP_NON_EXISTENT, msg)
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)
                LOG.info('use repostory the same with volumes other backup plan first[name:%s][repository:%s]!' % (
                    exist_plan_list[0].get("NAME"), exist_plan_list[0].get("NAMESPACENAME")))
            else:
                if copyplan_source_repository:
                    for repository in rep_meta_data:
                        if repository.get("NAME") == copyplan_source_repository:
                            rep_data = repository
                    if not rep_data:
                        msg = 'source Storage repository Not Found.'
                        error_msg = convert_error(ErrorCode.REP_NON_EXISTENT, msg)
                        LOG.error(error_msg)
                        raise exception.InvalidBackup(error_msg)
                    LOG.info('use repostory the same with copy plan source repository first[repository:%s]!' %
                             copyplan_source_repository)
                else:
                    rep_data = rep_meta_data[0]
                    for repository in rep_meta_data:
                        if int(repository.get("FREECAPACITY", "0")) >= int(rep_data.get("FREECAPACITY", "0")):
                            rep_data = repository
                            LOG.info('repository choice:%s,%s' % (rep_data.get("NAME"), rep_data.get("FREECAPACITY")))
                    LOG.info(
                        'not found any backupplan with this volume.use repostory with biggest free capacity first[%s]!' %
                        rep_data["NAME"])
            exist_pool_id = rep_data["PARENTID"]
            data_layout = self._get_data_layout_by_pool(exist_pool_id)
            policy_name = self._get_policy_name(data_layout)
            object_name["repository_name"] = rep_data["NAME"]
            object_name["policy_name"] = policy_name
            ebkconn = EbackupConnectionClient(self.__get_connection())
            ret, err, desc = ebkconn.create_backup_plan(object_name)
            if err:
                msg = self._convert_error(err, "create backup plan fail")
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            prepare_end = (source_volume_id + "Prepare backup finished.")
            LOG.info(prepare_end)
        else:
            unit_id, unit_filesystem_type, pool_id, pool_name = self._get_storage_info(backup)
            if unit_id is None:
                msg = 'Storage unit id is None.'
                error_msg = convert_error(ErrorCode.BRICK_NON_EXISTENT, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            if ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD:
                repository_name = repository_name + "_" + pool_name
                object_name["repository_name"] = repository_name
            b_rep_exist = self._be_exist(
                self.ebackup_manager_repository_list_uri, repository_name)
            if b_rep_exist == 0:
                LOG.info('the repository dose not exist, need to create it.')
                fre_cap = self._check_unit_free_capacity(unit_id)
                ebkconn = EbackupConnectionClient(self.__get_connection())
                data, err, desc = ebkconn.create_storage_repository(repository_name, pool_id, pool_name, fre_cap)
                if err:
                    msg = self._convert_error(err, "create storage repository fail")
                    LOG.error(msg)
                    raise exception.InvalidBackup(msg)
            policy_name = self._get_policy_name_by_filesys_type(unit_filesystem_type)
            object_name["policy_name"] = policy_name
            ebkconn = EbackupConnectionClient(self.__get_connection())
            ret, err, desc = ebkconn.create_backup_plan(object_name)
            if err:
                msg = self._convert_error(err, "create backup plan fail")
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            prepare_end = (source_volume_id + " Prepare backup finished.")
            LOG.info(prepare_end)

    def _get_object_id_with_name(self, uri, name):
        objectid = None
        dataobject = {}
        result = self._get_object(uri, name)
        error_info = self._get_error(result)
        if error_info != 0:
            LOG.error('get object id with name failed, eBackup error num is: %d' % error_info)
            return objectid
        data_info = self._get_data(result)
        if len(data_info) != 0:
            for data in data_info:
                if data.get("NAME") == name:
                    objectid = data.get("ID")
                    dataobject = data
        return objectid, dataobject

    def _get_latest_backup_id(self, backup):
        # get latest backup id by volume id
        try:
            volume_id = backup.get('volume_id')
            metadata = json.loads(backup.get('service_metadata'))
            domain_type = cst.DOMAIN_TYPE_BACKUP
            if 'IT' in metadata:
                domain_type = metadata['IT']
            storage_type = cst.BACKUP_TYPE_VOL
            if 'ST' in metadata:
                storage_type = metadata['ST']

            db_return = self.__backup_get_by_volume_id(volume_id)
            # latest backup object
            tmp_latest_backup = None
            for backup_item in db_return:
                backup_status = backup_item.get('status')
                # if status is not restoring and avaliable, then continue
                if backup_status == cst.BACKUP_STATUS_CREATING or backup_status == cst.BACKUP_STATUS_ERROR:
                    msg = '[' + backup_item.get('id') + '] status is creating or error.'
                    LOG.info(msg)
                    continue

                json_service_metadata = backup_item.get('service_metadata')
                if json_service_metadata in (None, ''):
                    LOG.info('service_metadata is not exist in backup_item.')
                    continue

                cur_metadata = json.loads(json_service_metadata)
                cur_domain_type = cst.DOMAIN_TYPE_BACKUP
                if 'IT' in cur_metadata:
                    cur_domain_type = cur_metadata['IT']
                cur_storage_type = cst.BACKUP_TYPE_VOL
                if 'ST' in cur_metadata:
                    cur_storage_type = cur_metadata['ST']

                if cur_domain_type != domain_type or storage_type != cur_storage_type:
                    msg = ('[' + backup_item.get('id') + '] IT is %d, ST is %d' % (cur_domain_type, cur_storage_type))
                    LOG.info(msg)
                    continue

                if (backup_status == cst.BACKUP_STATUS_DELETING):
                    msg = ('[' + backup_item.get('id') + '] is current backup, will choose another one.')
                    LOG.info(msg)
                    continue
                # check if is the first param
                if tmp_latest_backup is None:
                    tmp_latest_backup = backup_item
                    continue
                # get latest backup
                if backup_item.get('created_at') > tmp_latest_backup.get('created_at'):
                    tmp_latest_backup = backup_item
            if None != tmp_latest_backup:
                latest_backup_id = tmp_latest_backup.get('id')
                msg = ('latest backup is:' + str(latest_backup_id))
                LOG.info(msg)
                return tmp_latest_backup
        except Exception as msg:
            LOG.error(msg)
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, 'get latest backup id failed')
            raise exception.InvalidBackup(error_msg)

    def get_backup_plan_list(self, volume_id, st, dec=False):
        if st == 0 and not dec:
            plan_name = ('openstack_plan_%s' % volume_id)
        elif st == 0 and dec:
            plan_name = ('openstack_plan_dec_%s' % volume_id)
        elif dec is True:
            plan_name = ('openstack_csbs_plan_dec_%s' % volume_id)
        else:
            plan_name = ('openstack_csbs_plan_%s' % volume_id)
        plan_filter = '?filter=name:' + plan_name
        result = self._get_object_by_filter(
            self.ebackup_manager_plan_list_uri, plan_filter)
        error_code = self._get_error(result)
        if error_code != 0:
            msg = self._convert_error(error_code)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        backup_plan_list = self._get_data(result)
        return backup_plan_list

    def get_replication_copy_plan_list(self, volume_id, st, dec=False):
        if st == cst.COPY_TYPE_SINGLE_VOL and not dec:
            plan_name = ('openstack_rep_plan_%s' % volume_id)
        elif st == cst.COPY_TYPE_SINGLE_VOL and dec:
            plan_name = ('openstack_rep_plan_dec_%s' % volume_id)
        elif dec is True:
            plan_name = ('openstack_csbs_rep_plan_dec_%s' % volume_id)
        else:
            plan_name = ('openstack_csbs_rep_plan_%s' % volume_id)
        plan_filter = '?filter=NAME:' + plan_name
        result = self._get_object_by_filter(self.ebackup_copy_plan_uri, plan_filter)
        error_code = self._get_error(result)
        if error_code != 0:
            msg = self._convert_error(error_code)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        copy_plan_list = self._get_data(result)
        return copy_plan_list

    def query_image_capacity(self, url):
        meta_result = self.__get_connection().get(url)
        meta_error = self._get_error(meta_result)
        if int(meta_error) != 0:
            LOG.error("query backup image return error:%s", meta_error)
            if meta_error == ErrorCode.EC_COM_DB_ERROR or meta_error == ErrorCode.BOMG_BACKUP_OBJ_ID_ILLEGAL:
                LOG.warn("this kind of error[%s] are treated as good ones" % meta_error)
                return meta_result, False
            self.error_info.set_error_info(meta_error, self._get_error_description(meta_result))
            msg = self._convert_error(meta_error)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        return meta_result, True

    def query_image_count(self, plan_name, volume_id, deletion=0):
        get_image_count_url = self.ebackup_image_uri + '/count?filter=ASSOCIATEBACKUPDOMAINNAME::' + \
                              plan_name + '%20and%20MACHINEGUID::' + volume_id + '%20and%20DELETION::' + str(deletion)
        count_result = self.__get_connection().get(get_image_count_url)
        meta_error = self._get_error(count_result)
        if meta_error != 0:
            self.error_info.set_error_info(meta_error, self._get_error_description(count_result))
            msg = self._convert_error(meta_error)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        meta_data = self._get_data(count_result)
        str_image_count = meta_data.get("COUNT", 0)
        LOG.info("image count:%s" % str_image_count)
        return int(str_image_count)

    def _backup_size_of_volume(self, volume_id, domain_type, st):
        if st in (None, ""):
            LOG.warn("service meta miss st which cause calcing size isn't accurate, "
                     "driver dosen't update backup size this time")
            return 0, False, False

        backup_plan_list = self.get_backup_plan_list(volume_id, st)
        if not backup_plan_list:
            backup_plan_list = []

        if not backup_plan_list:
            LOG.warn("no backup plan exist")
            return 0, True, True
        image_count = 0
        plan_id_list = ''
        for plan in backup_plan_list:
            plan_id_list = plan_id_list + plan['UUID'] + ','
            image_count += self.query_image_count(plan['NAME'], volume_id)

        is_all_chain_dec = False if image_count > 0 else True
        if not plan_id_list:
            LOG.info("backup plan size is 0, treat chain db size as 0")
            return 0, True, is_all_chain_dec

        plan_id_list = plan_id_list[:-1]
        backup_size_uri = self.ebackup_get_backup_size + '?filter=BACKUP_PLAN_LIST::' + plan_id_list + \
                          '%20and%20TYPE::' + str(domain_type)
        LOG.info('get backup size url: %s.' % backup_size_uri)
        meta_result, query_res = self.query_image_capacity(backup_size_uri)
        if query_res:
            meta_data = self._get_data(meta_result)
            total_size = int(meta_data[0].get("BACKUPCAPACITY", "0")) / cst.Ki
            LOG.info("total size:%s", total_size)
            return total_size, True, is_all_chain_dec
        else:
            return 0, False, is_all_chain_dec

    def _copy_size_of_volume(self, volume_id, domain_type, st, backup):
        if backup is None:
            msg = self._convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG,
                                      "input parameter wrong while get copy chain size.")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        if st is None:
            LOG.error("service meta miss st which cause calcing size isn't accurate, "
                      "driver dosen't update backup size this time")
            return 0, False, False
        is_all_chain_dec = False
        copy_plan_name_list = list()
        if domain_type == cst.DOMAIN_TYPE_COPY:
            copy_plan_name_list.append(self._get_copy_plan_name(backup))
            copy_plan_name_list.append(self._get_copy_plan_name(backup) + "_" + volume_id)
        else:
            copy_plan_list = self.get_replication_copy_plan_list(volume_id, st)
            if not copy_plan_list:
                copy_plan_list = []
            image_count = 0
            for plan in copy_plan_list:
                image_count += self.query_image_count(plan['NAME'], volume_id)
                copy_plan_name_list.append(plan['NAME'])

            is_all_chain_dec = False if image_count > 0 else True

        copy_plan_uuid_list = list()
        for copy_plan_name in copy_plan_name_list:
            plan = self._get_copy_plan_by_name(self.ebackup_copy_plan_uri, copy_plan_name)
            if plan:
                copy_plan_uuid_list.append(plan.get("UUID"))

        if len(copy_plan_uuid_list) == 0:
            LOG.warning("this copy plan of this copy image does not exist on ebk server, permit driver to delete it")
            return 0, True, True

        # 1st item represent size, 2nd represent get size success at least 1 time, 3rd indicate the dec flag
        statistics = [0, False, is_all_chain_dec]
        for copy_plan_uuid in copy_plan_uuid_list:
            backup_size_uri = self.ebackup_get_backup_size + '?filter=DOMAINGUID::' + copy_plan_uuid + \
                              '%20and%20VOLUMEID::' + volume_id + '%20and%20TYPE::' + str(domain_type)
            LOG.info('get copy size url: %s.' % backup_size_uri)
            meta_result, query_res = self.query_image_capacity(backup_size_uri)
            if query_res:
                meta_data = self._get_data(meta_result)
                statistics[0] += int(meta_data[0].get("BACKUPCAPACITY", "0")) / cst.Ki
                statistics[1] = True
        LOG.info("total size:%s, %s, %s", statistics[0], statistics[1], statistics[2])
        return statistics[0], statistics[1], statistics[2]

    def _get_total_chain_size(self, volume_id, domain_type, st, backup):
        if cst.DOMAIN_TYPE_BACKUP == domain_type:
            return self._backup_size_of_volume(volume_id, domain_type, st)
        else:
            return self._copy_size_of_volume(volume_id, domain_type, st, backup)

    def _update_backup_total_size(self, backup):
        try:
            volume_id = self._get_cascaded_volume_id(backup)
            backup_id = backup['id']
            backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_service_metadata = backup_in_database.get('service_metadata')
            metadata = json.loads(json_service_metadata)
            domain_type = cst.DOMAIN_TYPE_BACKUP
            if 'IT' in metadata:
                domain_type = metadata['IT']
        except Exception as msg:
            LOG.error(str(msg))
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, 'Parameter error')
            raise exception.InvalidBackup(error_msg)

        latest_backup_obj = self._get_latest_backup_id(backup)
        if latest_backup_obj is not None:
            volume_backup_size, update, ad_flag = self._get_total_chain_size(
                volume_id, domain_type, metadata.get("ST", None), backup)
            if update:
                try:
                    latest_backup_obj['object_count'] = volume_backup_size
                    self._update_backup(latest_backup_obj['id'], object_count=volume_backup_size)
                    ExpensesEvent.sync_backup_to_cascading(latest_backup_obj.get('display_name', ''), 'object_count',
                                                           volume_backup_size)
                    if ad_flag:
                        latest_backup_meta = Utils.transfer_json_to_dict(latest_backup_obj.get('service_metadata'))
                        self._change_value_by_keys(latest_backup_meta, AD=1)
                        latest_backup_obj['service_metadata'] = Utils.transfer_to_json_str(latest_backup_meta)
                        self._update_backup(latest_backup_obj['id'],
                                            service_metadata=Utils.transfer_to_json_str(latest_backup_meta))
                        ExpensesEvent.sync_backup_to_cascading(latest_backup_obj.get('display_name', ''),
                                                               'service_metadata',
                                                               Utils.transfer_to_json_str(latest_backup_meta))
                except Exception as umsg:
                    LOG.error('update lastest backup failed, maybe it is not exist now, %s!', umsg)
        else:
            LOG.info('latest backup id is None, it will not update backup size.')
        return latest_backup_obj

    def _get_backupurl_by_volume(self, volume_id):
        # check volume_id
        dict_backupurl = {}
        try:
            backup_list = self.__backup_get_by_volume_id(volume_id, True)
            for backup_item in backup_list:
                backup_status = backup_item.get("status")
                if backup_status == "error":
                    continue
                json_server_meta = backup_item.get("service_metadata")
                if json_server_meta is None:
                    continue

                srv_metadata = json.loads(json_server_meta)
                if "backupurl" in srv_metadata:
                    backupurl = srv_metadata.get("backupurl")
                    if backupurl is None:
                        continue
                elif "backupurl_tmp" in srv_metadata:
                    backupurl = srv_metadata.get("backupurl_tmp")
                    if backupurl is None:
                        continue
                else:
                    continue
                backupid = backup_item.get("id")
                dict_backupurl[backupurl] = backupid
            return dict_backupurl
        except Exception as msg:
            LOG.error('get backupurl failed, errormsg is: %s' % msg)
            return None

    def _change_backup_mode(self, openstack_mode):
        # default backup mode.
        ebackup_mode = cst.FULL_BACKUP
        if openstack_mode == 0:
            ebackup_mode = cst.FULL_BACKUP
        elif openstack_mode == 1:
            ebackup_mode = cst.INCREMENT_BACKUP
        return ebackup_mode

    def _change_openstack_mode(self, ebackup_mode):
        # default openstack mode.
        openstack_mode = 1
        if int(ebackup_mode) == cst.FULL_BACKUP:
            openstack_mode = 0
        elif int(ebackup_mode) == cst.INCREMENT_BACKUP:
            openstack_mode = 1
        return openstack_mode

    def _run_backup_plan(self, backup, volume_file, redo_request_id=None):
        ''' Run backup plan '''
        try:
            volumeid = self._get_cascaded_volume_id(backup)
            snapshot_id = volume_file['snapshot_id']
            flag = self._check_length(snapshot_id, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ('snapshot id length does not comply, error length:%d' % len(snapshot_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            display_name = backup['display_name']
            flag = self._check_length(display_name, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ('display name length does not comply, error length:%d' % len(display_name))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            size = backup['size']
            # volume size length must be larger than 0 and less than 12
            if (size is None) or (len(str(size)) > cst.VOLUME_SIZE_LENGTH):
                msg = ('volume size is wrong.size:%s' % size)
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, "volume size is wrong")
                LOG.error(msg)
                raise Exception(error_msg)
            volume_size = size * cst.Gi
            backup_id = backup['id']
            backup_type = self._change_backup_mode(volume_file['backup_type'])
            storage_type = volume_file['storage_type']
            if str(storage_type) not in (cst.STORAGE_TYPE_VRM_FUSION_STORAGE,
                                         cst.STORAGE_TYPE_KVM_FUSION_STORAGE,
                                         cst.STORAGE_TYPE_VRMVIMS,
                                         cst.STORAGE_TYPE_KVMV3,
                                         cst.STORAGE_TYPE_VRMV3,
                                         cst.STORAGE_TYPE_VRMV3RDM,
                                         cst.STORAGE_TYPE_VRM_FUSION_RDM):
                error_msg = convert_error(ErrorCode.BD_STORAGE_TYPE_NOT_SUPPORT, "storage type is error")
                LOG.error('storage type is error, error type:%s' % storage_type)
                raise Exception(error_msg)

            cur_snapshot_url = volume_file['snapshot_url']
            if cur_snapshot_url is None:
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, 'current snapshot url is None')
                LOG.error(error_msg)
                raise Exception(error_msg)

            parent_snapshot_url = volume_file.get('parent_snapshot_url', "")
            if parent_snapshot_url is None:
                parent_snapshot_url = ""

            region_name = ""
        except Exception as msg:
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        LOG.info(volumeid + ':begin run the backup plan.')
        plan_name = getattr(backup, cst.PLAN_NAME) \
            if getattr(backup, cst.PLAN_NAME, None) is not None else self._get_plan_name(backup)

        # get plan name from backup server
        ebk_client = EbackupConnectionClient(self.__get_connection())
        plan_list, ret_code, ret_msg = ebk_client.get_plan_by_name(plan_name)
        if ret_code != 0 or not plan_list or 'ID' not in plan_list[0].keys():
            error_msg = convert_error(ErrorCode.PLAN_NON_EXISTENT, "backup plan id is None.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        plan_id = plan_list[0].get('ID')
        if redo_request_id is None:
            RequestID = "%d" % hash("%s" % uuid.uuid1())
            if RequestID.startswith('-'):
                RequestID = RequestID[1:]
        else:
            LOG.info("run redo backup plan. redo_request_id is: %s" % redo_request_id)
            RequestID = redo_request_id

        LOG.info("RequestID : %s" % RequestID)

        run_plan_body = {
            "DISPLAY_NAME": "",
            "VOLUME_ID": volumeid,
            "CHANNEL_TYPE": cst.CHANNEL_TYPE,
            "STORAGE_TYPE": storage_type,
            "VOLUME_SIZE": volume_size,
            "SNAPSHOT_URL": cur_snapshot_url,
            "PARENT_SNAPSHOT_URL": parent_snapshot_url,
            "VOLUME_OFFSET": cst.VOLUME_OFFSET,
            "ID": str(plan_id),
            "TYPE": cst.PLAN_TYPE_ID,
            "BACKUPMODE": backup_type,
            "REQUEST_ID": RequestID,
            "REGION": str(region_name),
        }
        run_plan_body = dict(run_plan_body, **self.evs_body)
        source_vol = call_db_fun(self.db.volume_get, self.context, backup['volume_id'])
        if str(storage_type) in (cst.STORAGE_TYPE_VRM_FUSION_STORAGE,
                                 cst.STORAGE_TYPE_VRM_FUSION_RDM,
                                 cst.STORAGE_TYPE_KVM_FUSION_STORAGE):
            vol_offset_size_str = self.__get_volume_offset_size_from_provider_location(source_vol)
            if vol_offset_size_str not in (None, ''):
                vol_offset_size = int(vol_offset_size_str)
                run_plan_body["VOLUME_OFFSET_SIZE"] = vol_offset_size
        if source_vol.get('vendor_volume_driver_metadata', ''):
            try:
                driver_m = source_vol.get('vendor_volume_driver_metadata')
                driver_m_dict = {item['key']: item['value'] for item in driver_m}
                if driver_m_dict.get('xen_clock_offset'):
                    run_plan_body["VOLUME_OFFSET_SIZE"] = int(driver_m_dict.get('xen_clock_offset'))
                    LOG.info('get migrate offset %s' % driver_m_dict.get('xen_clock_offset'))
            except Exception as e:
                LOG.error('load json of vendor_driver_metadata failed %s!' % e)
        run_plan_body_info = copy.deepcopy(run_plan_body)
        run_plan_body_info['DATA_ENC_INFO']['VK'] = "******"
        run_plan_body_info['DATA_ENC_INFO']['DEK'] = "******"
        run_plan_body_str = volumeid + ':run plan body is:' + str(run_plan_body_info)
        LOG.info("run plan body:%s" % run_plan_body_str)

        run_plan_url = self.run_backup_plan_uri
        # write RequestID to db
        request_id_s = "R_%s" % RequestID
        # get backup by backup id.
        backup_result = call_db_fun(self.db.backup_get, self.context, backup_id)
        json_metadata = backup_result.get('service_metadata')
        metadata = json.loads(json_metadata)
        # write CMKID and VK to metadata
        self._change_value_by_keys(metadata, ebk_T_I=str(request_id_s),
                                   CMKID=str(self.evs_body['DATA_ENC_INFO']['CMKID']),
                                   VK=str(self.evs_body['DATA_ENC_INFO']['VK']))
        self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

        task_id = ''
        request_id = ''
        for i in range(0, cst.TASK_RETRY_COUNT):
            result = self.__get_connection().put(run_plan_url, run_plan_body)
            error_number = self._get_error(result)
            error_description = self._get_error_description(result)
            if error_number != 0:
                self.error_info.set_error_info(error_number, error_description)
                msg = self._convert_error(error_number, str(error_description))
                LOG.error(msg)
                raise exception.InvalidBackup(msg)

            result_data = self._get_data(result)
            request_id = result_data.get("REQUEST_ID")
            try:
                task_id = self._get_task_id(cst.TASKTYPE_BACKUP, request_id, volumeid)
                self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
                break
            except exception.InvalidBackup as msg:
                if i == (cst.TASK_RETRY_COUNT - 1):
                    LOG.error("task %s is not found", request_id)
                    raise exception.InvalidBackup(cst.VERIFY_TASK_NOT_FOUND)
                LOG.info("get backup task fail: %s, try again: %s", msg, i)
                continue

        self._change_value_by_keys(metadata, AT=0.0)
        self._change_value_by_keys(metadata, SS=0)
        if is_infiniBand_env and self.__fusionstorage_ip:
            pht_type = 1 if self.__get_connection().get_phy_network_type() == 'ib' else 0
            self._change_value_by_keys(metadata, IB=pht_type)
        self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

        while True:
            time.sleep(cst.TIME_SLEEP_60)
            task_infos = self.get_task_info(task_id)

            if not isinstance(task_infos, dict):
                msg = ("task : %s failure,failed to get task info" % task_id)
                LOG.info(msg)
                raise exception.InvalidBackup(msg)

            task_status, task_progress, task_details = self._get_task_progress_and_details(task_infos,
                                                                                           cst.TASKTYPE_BACKUP)
            LOG.info("backup task progress is:%s" % (task_progress + '%'))
            # backup failed
            if task_status == cst.TASK_STATUS_FAILED:
                msg = ("eBackup Backup job failed, job id is:" + str(task_id))
                fail_reason_str = self._get_task_detail(task_details, msg)
                LOG.error("Backup job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                raise exception.InvalidBackup(fail_reason_str)

            if task_progress == '0':
                task_progress = '1'
            self._change_value_by_keys(metadata, progress=str(task_progress))
            self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

            # speed
            ret_status, task_speed = self._get_task_details_speed(task_infos)
            if ret_status is False:
                msg = "SPEED in backup is null, verify failed."
                LOG.error(msg)
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                self.error_info.set_error_info(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)

            if -1 != task_speed:
                self._change_value_by_keys(metadata, AT=task_speed)
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

            # backup completed
            if task_status == cst.TASK_STATUS_COMPLETED:
                # spacesavingrate
                task_space_saving_rate = self._get_task_details_spacesavingrate(task_infos)
                if task_space_saving_rate in (None, ''):
                    msg = "SPACESAVINGRATE in backup is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)

                self._change_value_by_keys(metadata, SS=int(task_space_saving_rate))
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

                image_id = self._get_new_snap_uuid(task_id)
                if image_id is None:
                    # if image id is null, it means backup failed.
                    msg = 'get new image id is None'
                    error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, msg)
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)

                # Get SNAPTYPE and Update Type in service_metadata when task was completed.
                task_snaptype = task_infos.get("SNAPTYPE")
                if task_snaptype is None:  # old version
                    self._change_value_by_keys(metadata, backupurl=str(image_id), progress='0')
                else:
                    backup_mode = self._change_openstack_mode(task_snaptype)
                    self._change_value_by_keys(metadata, backupurl=str(image_id), progress='0', Type=backup_mode)

                # get volume backup size (totle size)
                volume_backup_size, update, ad_flag = self._get_total_chain_size(volumeid,
                                                                                 cst.DOMAIN_TYPE_BACKUP,
                                                                                 metadata.get("ST", None),
                                                                                 backup)
                if ad_flag:
                    self._change_value_by_keys(metadata, AD=1)
                self._change_value_by_keys(metadata, CS=int(volume_backup_size), ebk_T_I="",
                                           SNAPSHOTSIZE=task_infos.get("SNAPSHOTSIZE"))
                self._update_backup(backup_id,
                                    service_metadata=Utils.transfer_to_json_str(metadata),
                                    object_count=volume_backup_size)
                run_success = (
                                  "Run the backup plan success, Volume id:%s, Backup id:%s, Cinder request id:%s, eBackup request id:%s, Backupurl:%s, Task id:%s.") % (
                                  str(volumeid), backup_id, request_id_s, request_id, str(image_id), task_id)
                LOG.info(run_success)
                break

    def _get_new_snap_uuid(self, task_id):
        get_image_id_url = self.ebackup_image_uri + "?filter=JOBID::" + task_id + "&range=[0-15]&sortby=CREATETIME,d"
        snap_count = 0
        while True:
            meta_result = self.__get_connection().get(get_image_id_url)
            snap_count += 1
            meta_error = self._get_error(meta_result)
            if meta_error != 0:
                self.error_info.set_error_info(meta_error, self._get_error_description(meta_result))
                msg = self._convert_error(meta_error)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            meta_data = self._get_data(meta_result)
            if len(meta_data) != 0:
                image_id = meta_data[0].get("ID")
                break
            time.sleep(cst.TIME_SLEEP_20)
            if snap_count < cst.QUERY_COUNT:
                LOG.info('try get new image id, get count:%s' % snap_count)
                continue
            else:
                LOG.error("get new image id is failed")
                return None

        LOG.info("get new image id is: %s" % image_id)
        return image_id

    def _get_image_by_id(self, image_id, sync=False):
        get_image_uri = self.ebackup_manager_single_snap + '?id=' + image_id
        image = None
        result = self.__get_connection().get(get_image_uri, sync)
        error_number = self._get_error(result)
        if error_number == ErrorCode.IMAGE_NON_EXISTENT:
            LOG.info('image does not exist.')
            return image
        elif error_number != 0:
            msg = 'query image info by id is fail'
            self.error_info.set_error_info(error_number, self._get_error_description(result))
            error_msg = convert_error(error_number, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        image = self._get_data(result)
        return image

    def _get_copy_snaps(self, src_snap_id):
        get_image_copy_uri = self.ebackup_image_copy_uri
        get_image_copy_body = {
            "ID": src_snap_id,
            "TYPE": IMAGE_OBJ_ID
        }
        copy_images = None
        request_str = 'get copy images body is:' + str(get_image_copy_body)
        LOG.info(request_str)
        result = self.__get_connection().put(get_image_copy_uri, get_image_copy_body)
        error_number = self._get_error(result)
        if error_number == ErrorCode.IMAGE_NON_EXISTENT:
            LOG.info('copy image does not exist.')
            return copy_images
        elif error_number != 0:
            self.error_info.set_error_info(error_number, self._get_error_description(result))
            msg = self._convert_error(error_number, "get copy image fail")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        result_data = self._get_data(result)
        response_str = 'get copy images response is:' + str(result_data)
        LOG.info(response_str)
        copy_images_str = result_data.get("COPYSNAPS")
        copy_images = json.loads(copy_images_str)
        if copy_images is None:
            LOG.info('copy images is none')
        return copy_images

    def _get_copy_snap_guids_by_snap_id(self, image_id, volume_id, need_check_canbecopy=False):
        copy_images = self._get_copy_snaps(image_id)
        copy_snap_guid_list = []
        dict_backupurl = []
        srv_meta = {}
        if need_check_canbecopy:
            dict_backupurl = self._get_backupurl_by_volume(volume_id)
            if dict_backupurl is None:
                return copy_snap_guid_list, srv_meta
        if copy_images is not None:
            for copy_image in copy_images:
                LOG.info('get copy image ' + str(copy_image) + ', need_check_canbecopy:' + str(need_check_canbecopy))
                copy_snap_guid = copy_image["TARGETSNAPGUID"]
                if need_check_canbecopy:
                    if copy_snap_guid in dict_backupurl:
                        LOG.info("update backup[%s]'s backupurl to empty", dict_backupurl[copy_snap_guid])
                        copy_snap_guid_list.append(copy_snap_guid)
                        tmp_context = self.context.elevated(read_deleted='yes')
                        backup = call_db_fun(self.db.backup_get, tmp_context, dict_backupurl[copy_snap_guid])
                        srv_meta = Utils.transfer_json_to_dict(backup.get("service_metadata"))
                        if 'backupurl' in srv_meta:
                            srv_meta['backupurl'] = ""
                            srv_meta['ebk_T_I'] = ""
                            call_db_fun(self.db.backup_update, tmp_context, dict_backupurl[copy_snap_guid],
                                        {'object_count': 0, 'service_metadata': Utils.transfer_to_json_str(srv_meta)})
                        continue
                    canbecopy = copy_image.get("CANBECOPY")
                    LOG.info("the copy image id is: %s, CANBECOPY is: %s" % (str(copy_snap_guid), str(canbecopy)))
                    copy_snap_guid_list.append(copy_snap_guid)
                else:
                    copy_snap_guid_list.append(copy_snap_guid)
        return copy_snap_guid_list, srv_meta

    def _delete_object(self, object_uri, object_id, body):
        delete_object_uri = str(object_uri) + '/' + str(object_id)
        delete_result = self.__get_connection().delete(delete_object_uri, body)
        self.error_info.set_error_info(self._get_error(delete_result), self._get_error_description(delete_result))
        return delete_result

    def _delete_image_object(self, object_uri, object_id, requestID, body, retry=True, sync=False):
        delete_object_uri = str(object_uri) + '/' + str(object_id) + '?REQUEST_ID=' + str(requestID)
        delete_result = self.__get_connection().delete(delete_object_uri, body, retry, sync)
        self.error_info.set_error_info(self._get_error(delete_result), self._get_error_description(delete_result))
        return delete_result

    def _do_delete(self, backup, last_status):
        """Begin hwsebackup delete."""
        try:
            backup_id = backup['id']
            backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_service_metadata = backup_in_database.get('service_metadata')
            metadata = json.loads(json_service_metadata)
        except Exception as msg:
            LOG.error(str(msg))
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, 'Parameter error')
            raise exception.InvalidBackup(error_msg)

        image_id = metadata.get('backupurl')
        task_ebk = metadata.get('ebk_T_I')
        if (image_id in ('', None)) and (task_ebk not in (None, '')):
            task_id, task_type = self._get_task_id_and_task_type(str(task_ebk))
            if task_id is None or task_type is None:
                LOG.info('Get task id and task type failed. ebk_T_I=%s' % str(task_ebk))
                return
            if int(task_type) == cst.TASKTYPE_COPY:
                LOG.info('The task(%s) is copy, but image id is null, so no need delete from eBackup.' % str(task_ebk))
                return
            image_id = self._get_image_by_task(task_id)

        if image_id == ErrorCode.TSKMG_TASK_ID_NOT_EXIST:
            LOG.info('The task(%s) does not exist.' % str(task_ebk))
            return
        del_image_msg = self._delete_image(backup, str(image_id), last_status)
        if del_image_msg == 1:
            return

    def _get_task_id_and_task_type(self, task_ebk_T_I):
        if task_ebk_T_I.startswith('R_'):
            request_id = task_ebk_T_I[2:]
            task_url = self.ebackup_progress_uri + "?filter=REQUEST_ID::" + str(
                request_id) + "&range=[0-15]" + '&sortby=CREATETIME,d'
            LOG.info(task_url)
            meta_result = self.__get_connection().get(task_url)
            meta_error = self._get_error(meta_result)
            if meta_error != 0:
                self.error_info.set_error_info(meta_error, self._get_error_description(meta_result))
                msg = self._convert_error(meta_error, 'get task id and task type by request id failed.')
                LOG.error(msg)
                raise exception.InvalidBackup(msg)

            meta_data = self._get_data(meta_result)
            if len(meta_data) != 0:
                task_id = meta_data[0].get("ID")
                task_type = meta_data[0].get("TASKTYPE")
                return task_id, task_type

            LOG.info('The task(RequestID:%s) does not exist.' % str(task_ebk_T_I))
            return None, None

        task_url = self.ebackup_task_uri + "/get_single_jobs?id=%s" % task_ebk_T_I
        LOG.info(task_url)
        meta_result = self.__get_connection().get(task_url)
        error_number = self._get_error(meta_result)
        if error_number == ErrorCode.TSKMG_TASK_ID_NOT_EXIST:
            LOG.info('The task(ID:%s) does not exist.' % str(task_ebk_T_I))
            return None, None

        if error_number != 0:
            self.error_info.set_error_info(error_number, self._get_error_description(meta_result))
            msg = self._convert_error(error_number, "get task type by task id failed")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        meta_data = self._get_data(meta_result)
        task_type = meta_data.get("TASKTYPE")
        task_id = task_ebk_T_I
        return task_id, task_type

    def _is_image_migrate(self, image_id):
        try:
            image = self._get_image_by_id(image_id)
            if image and image.get('VALIDITY') == '3':
                return True
        except Exception:
            LOG.exception("Failed to get image: {}.".format(image_id))
        return False

    def _delete_image(self, backup, image_id, last_status):
        ''' delete backup image '''
        backup_id = backup['id']
        backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
        json_service_metadata = backup_in_database.get('service_metadata')
        metadata = json.loads(json_service_metadata)
        evs_body = self.evs_body
        evs_body['REGION'] = ""
        is_migrated = self._is_image_migrate(image_id)
        if metadata.get('IT') == cst.IMPORT_TYPE_NATIVE_TO_CLOUD or \
                is_migrated:
            # If the backup has been migrated, storage repository information
            # is required for the delete task.
            en_uds_login_ak = ebackup_driver_conf.ebackup_storage_username
            en_uds_login_sk = ebackup_driver_conf.ebackup_storage_password
            uds_login_ak = Utils.decrypt_password(en_uds_login_ak)
            uds_login_sk = Utils.decrypt_password(en_uds_login_sk)
            import_snap_data = dict()
            import_snap_data['BACKUPSTORAGETYPE'] = \
                int(metadata.get('BT', '7'))
            import_snap_data['BACKUPDATALAYOUT'] = metadata.get('DL')
            storage_path = metadata.get('BP') if is_migrated else \
                metadata.get('BP') + '/' + metadata.get('SP')
            import_snap_data['BACKUPSTORAGEPATH'] = storage_path
            import_snap_data['BACKUPSTORAGEUSERNAME'] = uds_login_ak
            import_snap_data['BACKUPSTORAGEPASSWD'] = uds_login_sk
            evs_body['IMPORTSNAPDATA'] = import_snap_data
            evs_body['DELETEIMPORTSNAP'] = 0 if is_migrated else 1
            log_body = copy.deepcopy(evs_body)
            log_body['DATA_ENC_INFO']['DEK'] = '***'
            log_body['IMPORTSNAPDATA']['BACKUPSTORAGEPASSWD'] = '***'
            LOG.info('The deleted body is %s.' % (json.dumps(log_body)))

        if last_status == 'pre_delete':
            evs_body['DELETESTATUS'] = last_status
        if image_id not in (None, '', 'None'):
            task_id = None
            RID = metadata.get('ebk_T_D')
            if RID not in(None, '', 'None'):
                task_id, task_type = self._get_task_id_and_task_type(str(RID))
            if task_id is None or task_type is None:
                LOG.info('Get task id failed. ebk_T_D=%s' % str(RID))
                RID = "%d" % hash("%s" % uuid.uuid1())
                if RID.startswith('-'):
                    RID = RID[1:]
                LOG.info("generate new RequestID : %s" % RID)
                request_id_s = "R_%s" % RID
                self._change_value_by_keys(metadata, ebk_T_D=str(request_id_s))
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
                dslt = self._delete_image_object(self.ebackup_image_uri,
                                                 image_id, RID, evs_body)
                error_number = self._get_error(dslt)
                if error_number == ErrorCode.IMAGE_NON_EXISTENT:
                    LOG.info('(ID:%s) was deleted.' % (str(image_id)))
                    return
                elif error_number != 0:
                    error_msg = convert_error(error_number,
                                              'delete image failed')
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)
                else:
                    del_data = self._get_data(dslt)
                    # if in asyc delete, return
                    is_asyc_delete = del_data.get("ISASYNC", None)
                    if is_asyc_delete == '1':
                        LOG.info("async del image success,%s", is_asyc_delete)
                        return
                    else:
                        LOG.info("not do async delete,%s", is_asyc_delete)

                    request_id = del_data.get("REQUEST_ID")
                    task_id = self._get_task_id(cst.TASKTYPE_DEL_IMAGE, request_id, self._get_cascaded_volume_id(backup))

            self._change_value_by_keys(metadata,
                                       ebk_T_D=str(task_id), ebk_T_I="")
            s_meta = Utils.transfer_to_json_str(metadata)
            self._update_backup(backup_id, service_metadata=s_meta)
            LOG.info("delete image task begin.")
            while True:
                time.sleep(cst.TIME_SLEEP_30)
                task_infos = self.get_task_info(task_id)

                if not isinstance(task_infos, dict):
                    LOG.info("task : %s failure,failed to get task info" % task_id)
                    break

                task_status, task_progress, tdetls = \
                    self._get_task_progress_and_details(task_infos,
                                                        cst.TASKTYPE_DEL_IMAGE)
                LOG.info('delete image task progress is:%s' % task_progress)
                if task_status == cst.TASK_STATUS_FAILED:
                    msg = ("eBackup Delete backup image job failed, job id is:" + str(task_id))
                    if tdetls['code'] == str(ErrorCode.IMAGE_DELETING):
                        task_id = tdetls['params'][0]
                        self._change_value_by_keys(metadata,
                                                   ebk_T_D=str(task_id))
                        s_meta = Utils.transfer_to_json_str(metadata)
                        self._update_backup(backup_id,
                                            service_metadata=s_meta)
                        continue
                    elif tdetls['code'] == str(ErrorCode.IMAGE_NON_EXISTENT):
                        LOG.info('The (image %s) was deleted' % str(image_id))
                        return
                    fail_reason_str = self._get_task_detail(tdetls, msg)
                    LOG.error("Delete backup image job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                    self._change_value_by_keys(metadata, ebk_T_D='')
                    self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
                    raise exception.InvalidBackup(fail_reason_str)

                self._change_value_by_keys(metadata, progress=str(task_progress))
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

                if task_status == cst.TASK_STATUS_COMPLETED:
                    self._change_value_by_keys(metadata,
                                               progress='0',
                                               backupurl='',
                                               ebk_T_D='')
                    self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
                    break
            LOG.info("delete image task success.")
        else:
            LOG.info("The image id of backup(%s) is null, no need to delete it." % str(backup_id))

    def _check_length(self, obj, length):
        ''' check object length, does not comply return True'''
        if obj is None:
            return False
        if len(obj) > length:
            return True
        else:
            return False

    def _get_copy_mode(self, backup):
        display_desc = backup['display_description']
        if display_desc in (None, ''):
            return -1
        try:
            json_display_desc = Utils.transfer_json_to_dict(display_desc)
            backup_mode = json_display_desc.get('backup_mode')
        except Exception as msg:
            LOG.info("get backup_mode from description failed, invalid json format.")
            LOG.error(msg)
            return -1
        if backup_mode in (None, ''):
            LOG.error('backup mode is none.')
            return -1
        if backup_mode != BACKUP_MODE_LOCAL and backup_mode != BACKUP_MODE_TO_CLOUD:
            LOG.error('backup mode is not local and cloud.')
            return -1
        return backup_mode

    def __get_backup_metadata(self, backup):
        if 'service_metadata' in backup:
            metadata = backup.get('service_metadata')
            if metadata not in ("", None):
                return Utils.transfer_json_to_dict(metadata)
            return {}
        return {}

    def __get_snapshot_url(self, storage_type, volume, snapshot_id, is_parent):
        if snapshot_id in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "snapshot_id is none.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        # parent snapshot id: BACKUPS--service_metadata--snap_id, backup is last backup record
        # current snapshot id: VOLUMES--snapshot_id, volume is a link clone volume that created from snapshot
        snapshot_url = None
        source_snapshot = call_db_fun(self.db.snapshot_get, self.context, snapshot_id)
        if cst.STORAGE_TYPE_VRM_FUSION_STORAGE == storage_type or cst.STORAGE_TYPE_VRM_FUSION_RDM == storage_type:
            # VRM+FusionStorage: snapshot_id is a openstack snapshot id, should convert to VRM format
            snapshot_id_fc = str(snapshot_id).replace('-', '')
            dsw_manager_ip, pool_id = self.__get_ip_and_pool_from_provider(source_snapshot)
            snapshot_name = self.__get_snap_name_from_provider_location(source_snapshot)
            if dsw_manager_ip in (None, '') or pool_id in (None, '') or snapshot_name in (None, ''):
                dsw_manager_ip, pool_id = self.__get_vrm_fusionstorage_manager_ip_and_pool_id(volume)
                snapshot_url = 'http://' + dsw_manager_ip + '/' + str(pool_id) + '/' + snapshot_id_fc
            else:
                snapshot_url = 'http://' + dsw_manager_ip + '/' + str(pool_id) + '/' + snapshot_name
        elif cst.STORAGE_TYPE_KVM_FUSION_STORAGE == storage_type:
            # KVM+FusionStorage: snapshot_id is a openstack snapshot id, should convert to FusionStorage format
            dsw_manager_ip, pool_id = self.__get_ip_and_pool_from_provider(source_snapshot)
            snapshot_name = self.__get_snap_name_from_provider_location(source_snapshot)
            if dsw_manager_ip in (None, '') or pool_id in (None, '') or snapshot_name in (None, ''):
                dsw_manager_ip, pool_id = self.__get_kvm_fusionstorage_manager_ip_and_pool_id(volume)
                snapshot_url = 'http://' + dsw_manager_ip + '/' + str(pool_id) + '/' + 'snapshot-' + snapshot_id
            else:
                snapshot_url = 'http://' + dsw_manager_ip + '/' + str(pool_id) + '/' + snapshot_name
        elif cst.STORAGE_TYPE_KVMV3 == storage_type:
            # get sn and snapshotId from provider_auth field prior, then enter original branch if not exist.
            storage_sn, lun_snapshot_id = self.__get_v3_sn_and_lunid_from_snapshots(source_snapshot)
            if storage_sn in (None, ''):
                storage_sn = self.__get_v3_sn(volume)
            if is_parent:
                # KVM+V3: parent snapshot id is a real snapshot id from v3 storage
                snapshot_url = 'https://' + storage_sn + '/' + snapshot_id
            else:
                # KVM+V3: current snapshot id is a opensatck snapshot id, should get real v3 snapshot id from SNAPSHOTS--provider_location
                if lun_snapshot_id in (None, ''):
                    lun_snapshot_id = source_snapshot['provider_location']
                snapshot_url = 'https://' + storage_sn + '/' + lun_snapshot_id
        elif cst.STORAGE_TYPE_VRMV3 == storage_type or cst.STORAGE_TYPE_VRMV3RDM == storage_type:
            # get sn and snapshotId from provider_auth field prior, then enter original branch if not exist.
            # VRM+V3: snapshot_id is a opensatck snapshot id, should get real v3 snapshot id from SNAPSHOTS--provider_location
            snapshot_uuid = uuid.UUID(
                snapshot_id)  # VRM Cinder driver may replace '-' in J version, so convert to openstack uuid
            source_snapshot = call_db_fun(self.db.snapshot_get, self.context, str(snapshot_uuid))
            storage_sn, lun_snapshot_id = self.__get_v3_sn_and_lunid_from_snapshots(source_snapshot)
            if storage_sn in (None, ''):
                storage_sn = self.__get_v3_sn(volume)
            try:
                if lun_snapshot_id in (None, ''):
                    lun_snapshot_id = source_snapshot['provider_location']
                snapshot_url = 'https://' + storage_sn + '/' + lun_snapshot_id
            except:
                LOG.error("snapshot %s or provider_location not exit", snapshot_uuid)
        else:
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "storage type not support.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        return snapshot_url

    def get_volume_image_metadata(self, context, volume):
        db_data = call_db_fun(self.db.volume_glance_metadata_get, context, volume['id'])
        LOG.info("Get volume image-metadata completed successfully.")
        return {meta_entry.key: meta_entry.value for meta_entry in db_data}

    def __get_volume_bootable(self, vol_second_os, vol_first_id):
        volume_bootable = False
        vol_first_os = call_db_fun(self.db.volume_get, self.context, vol_first_id)
        try:
            vol_image_meta = self.get_volume_image_metadata(self.context, vol_second_os)
            if vol_image_meta:
                LOG.info('The bootable is True.')
                volume_bootable = True

        except Exception as ex:
            LOG.info('get_volume_image_metadata is error [%s]' % ex)

        if vol_first_os.get('bootable') is True:
            LOG.info('The bootable is True.')
            volume_bootable = True

        return volume_bootable

    def __get_none_clone_volume_bootable(self, vol_first_id):
        volume_bootable = False
        vol_first_os = call_db_fun(self.db.volume_get, self.context, vol_first_id)

        if vol_first_os.get('bootable') is True:
            LOG.info('The bootable is True.')
            volume_bootable = True

        return volume_bootable

    def __get_backup_volume_type(self, volume_id, volume_file):
        vol_type_name = ""
        volume = call_db_fun(self.db.volume_get, self.context, volume_id)
        vol_type_id = volume['volume_type_id']
        if vol_type_id != None and vol_type_id != "":
            volume_type = call_db_fun(self.db.volume_type_get, self.context, vol_type_id)
            if volume_type != None:
                vol_type_name = volume_type['name']
        volume_file['volume_type_name'] = vol_type_name

    def __get_backup_volume_file(self, backup):
        """Get volume file object representing the volume if not supply this information."""

        volume_file = {}
        link_clone_vol = call_db_fun(self.db.volume_get, self.context, backup['volume_id'])
        LOG.info('a backup for volume %s, id : %s.' % (link_clone_vol['name'], backup['volume_id']))

        # Get AZ name
        self.__availability_zone = link_clone_vol['availability_zone']

        # Get storage type.
        volume_storage_type = self.__get_volume_storage_type(link_clone_vol)
        volume_file['storage_type'] = int(volume_storage_type)

        snapshot_id = link_clone_vol['snapshot_id']
        source_snapshot = call_db_fun(self.db.snapshot_get, self.context, snapshot_id)
        source_volume_id = source_snapshot['volume_id']

        backup_list = self.__backup_get_by_volume_id(source_volume_id)
        last_backup = self.__get_last_backup(backup_list, False, "", "backup")

        # Get backup mode from inc
        if self._get_backup_mode_from_inc(backup) is cst.BACKUP_CMD_FULL:
            # do full backup
            LOG.info('Do full backup.')
            last_backup = None
        else:
            LOG.info('INC is None or INC is 1.')

        # Get backup type, parent_id, and parent_snapshot_url, do Full_backup or Incremental_backup.
        if not last_backup:
            volume_file['backup_type'] = 0
            volume_file['parent_id'] = None
            volume_file['parent_snapshot_url'] = None
        else:
            parent_snapshot_id = last_backup.get('container')
            if parent_snapshot_id is None:
                parent_snapshot_id = last_backup.get('snapshot_id')
            LOG.info("Get last_backup's snapshot_id:%s" % parent_snapshot_id)

            if parent_snapshot_id:
                try:
                    parent_snapshot = call_db_fun(self.db.snapshot_get, self.context, parent_snapshot_id)
                    LOG.info('parent snapshot:%s' % parent_snapshot)
                except:
                    LOG.info('parent snapshot %s does not exist, it may have been deleted.' % parent_snapshot_id)
                    parent_snapshot = None
            else:
                parent_snapshot = None

            if parent_snapshot is None:
                LOG.info('No parent snapshot found, do full backup.')
                volume_file['backup_type'] = 0
                volume_file['parent_id'] = None
                volume_file['parent_snapshot_url'] = None
            else:
                LOG.info('parent snapshot found, do incremental backup.')
                volume_file['backup_type'] = 1
                volume_file['parent_id'] = last_backup['id']
                # Get last backup snapshot_url.
                volume_file['parent_snapshot_url'] = self.__get_snapshot_url(volume_storage_type, link_clone_vol,
                                                                             parent_snapshot_id, True)

        # Get current backup snapshot_url.
        snapshot_url = self.__get_snapshot_url(volume_storage_type, link_clone_vol, snapshot_id, False)
        volume_file['snapshot_url'] = snapshot_url

        # Get snapshot id
        if cst.STORAGE_TYPE_KVMV3 == volume_storage_type:
            # KVM+V3: BACKUPS--service_metadata--snap_id is a real snapshot id of v3 storage
            # eBackup driver will update snapshot_id to BACKUPS--service_metadata--snap_id
            # upgrade from J version to M version, should keep the historical implementation
            volume_file['snapshot_id'] = snapshot_url.split('/').pop().strip()
        else:
            # Others: a openstack snapshot id
            volume_file['snapshot_id'] = snapshot_id

        # Get source volume id
        volume_file['source_volume_id'] = source_volume_id

        # Set default value to reserved parameters
        volume_file['bootable'] = self.__get_volume_bootable(link_clone_vol, source_volume_id)
        volume_file['image_id'] = None
        volume_file['clone_volume_url'] = None

        LOG.info("The volume_file [%s] is constructed", volume_file)
        return volume_file

    # --- Begin M Version Snapshot Backup Support 2016/12/12 ---
    def _get_backup_mode_from_inc(self, backup):
        display_desc = backup['display_description']
        if display_desc in (None, ''):
            LOG.info('display_description is None.')
            return -1
        try:
            json_display_desc = Utils.transfer_json_to_dict(display_desc)
            inc = json_display_desc.get('INC')
        except Exception as msg:
            LOG.info(msg)
            return -1
        if inc in (None, ''):
            LOG.info('INC is none.')
            return -1
        if (inc != cst.BACKUP_CMD_FULL and inc != cst.BACKUP_CMD_INC):
            LOG.info('INC is not 0 and 1.')
            return -1

        LOG.info('INC is %d', inc)
        return inc

    def _get_cloud_service_type(self, backup):
        display_desc = backup['display_description']
        service = 'CSBS'
        try:
            if display_desc not in (None, ''):
                json_display_desc = Utils.transfer_json_to_dict(display_desc)
                inc = json_display_desc.get('CBR', '')
                if inc == 1:
                    service = 'CBR'
        except Exception as msg:
            LOG.error(msg)
        finally:
            LOG.info('Cloud Service Type %s' % service)
        return service

    def set_fusion_storage_ip_for_connection(self, volume_file, task_type):
        if is_infiniBand_env:
            try:
                storage_type = str(volume_file['storage_type'])
                LOG.info("fusion storage storage_type:%s", storage_type)
                if (storage_type == cst.STORAGE_TYPE_VRM_FUSION_STORAGE or
                            storage_type == cst.STORAGE_TYPE_KVM_FUSION_STORAGE or
                            storage_type == cst.STORAGE_TYPE_VRM_FUSION_RDM):
                    url = volume_file['snapshot_url'] if task_type == cst.TASKTYPE_BACKUP else volume_file['volume_url']
                    LOG.info("fusion storage url:%s", url)
                    fusionstorage_info = url.split("/")
                    if len(fusionstorage_info) < 3:
                        LOG.info("get fusionSphere ip failed, set __fusionstorage_ip failed")
                        return
                    self.__fusionstorage_ip = fusionstorage_info[2]
                LOG.info("fusion storage ip:%s", self.__fusionstorage_ip)
            except Exception:
                LOG.info("fetch __fusionstorage_ip failed")
                raise exception.InvalidBackup("fetch __fusionstorage_ip failed")

    def get_dec_flag(self, backup):
        des = Utils.transfer_json_to_dict(backup.get("display_description"))
        return des.get('DEC', 0)

    def __get_backup_snapshot_volume_file(self, backup):
        """Get volume file object representing the volume if not supply this information."""

        volume_file = {}
        source_vol = call_db_fun(self.db.volume_get, self.context, backup['volume_id'])
        LOG.info('a snapshot backup for volume %s, id : %s.' % (source_vol['name'], backup['volume_id']))

        # Get AZ name
        self.__availability_zone = source_vol['availability_zone']
        volume_file['availability_zone'] = source_vol['availability_zone']
        # Get storage type.
        volume_storage_type = self.__get_volume_storage_type(source_vol)
        volume_file['storage_type'] = int(volume_storage_type)

        # Get volume owner machineID
        description = Utils.transfer_json_to_dict(backup.get("display_description"))
        volume_file['machine_id'] = description.get("VMID", "")
        volume_file['dec'] = description.get('DEC', 0)

        # Get snapshot id from backup
        snapshot_id = backup['snapshot_id']
        LOG.info('Got snapshot_id from backup.')

        source_snapshot = call_db_fun(self.db.snapshot_get, self.context, snapshot_id)
        source_volume_id = source_snapshot['volume_id']
        backup_list = call_db_fun(self.db.backup_get_by_volume_id, self.context, source_volume_id)
        last_backup = self.__get_last_backup(backup_list, True, backup['id'], "backup")

        force_full_backup = False
        if last_backup and ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD:
            last_metadata = json.loads(last_backup.get('service_metadata'))
            last_backup_storage_type = last_metadata.get('BT')
            last_backup_datalayout = last_metadata.get('DL')
            current_conf_datalayout = last_backup_datalayout
            if last_backup_storage_type == FILESYS_CIFS:
                current_conf_datalayout = int(ebackup_driver_conf.ebackup_cifs_data_layout)
            elif last_backup_storage_type == FILESYS_NFS:
                current_conf_datalayout = int(ebackup_driver_conf.ebackup_nfs_data_layout)
            elif last_backup_storage_type == FILESYS_S3:
                current_conf_datalayout = int(ebackup_driver_conf.ebackup_uds_data_layout)
            elif last_backup_storage_type == FILESYS_SAN:
                current_conf_datalayout = int(ebackup_driver_conf.ebackup_san_data_layout)
            if current_conf_datalayout != last_backup_datalayout:
                force_full_backup = True
                LOG.info('Do full backup.Because chain changed,datalayout changed!')

        # Get backup mode from inc
        if self._get_backup_mode_from_inc(backup) is cst.BACKUP_CMD_FULL or force_full_backup:
            # do full backup
            LOG.info('Do full backup.')
            last_backup = None
        else:
            LOG.info('INC is None or INC is 1.')

        # Get backup type, parent_id, and parent_snapshot_url, do Full_backup or Incremental_backup.
        if not last_backup:
            volume_file['backup_type'] = 0
            volume_file['parent_id'] = None
            volume_file['parent_snapshot_url'] = None
        else:
            parent_snapshot_id = last_backup.get('snapshot_id')
            LOG.info("Get last_backup's snapshot_id:%s" % parent_snapshot_id)
            try:
                parent_snapshot = call_db_fun(self.db.snapshot_get, self.context, parent_snapshot_id)
                LOG.info('Parent snapshot:%s' % parent_snapshot)
            except:
                LOG.info('Parent snapshot %s does not exist, it may have been deleted.' % parent_snapshot_id)
                parent_snapshot = None

            if not parent_snapshot:
                LOG.info('No parent snapshot found, do full backup.')
                volume_file['backup_type'] = 0
                volume_file['parent_id'] = None
                volume_file['parent_snapshot_url'] = None
            else:
                LOG.info('Current snapshot and parent snapshot exist, do incremental backup.')
                volume_file['backup_type'] = 1
                volume_file['parent_id'] = last_backup['id']
                # Get last backup snapshot_url.
                volume_file['parent_snapshot_url'] = self.__get_snapshot_url(volume_storage_type, source_vol,
                                                                             parent_snapshot_id, True)
        # Get current backup snapshot_url.
        snapshot_url = self.__get_snapshot_url(volume_storage_type, source_vol, snapshot_id, False)
        volume_file['snapshot_url'] = snapshot_url

        # Get snapshot id
        if cst.STORAGE_TYPE_KVMV3 == volume_storage_type:
            # KVM+V3: BACKUPS--service_metadata--snap_id is a real snapshot id of v3 storage
            # eBackup driver will update snapshot_id to BACKUPS--service_metadata--snap_id
            # upgrade from J version to M version, should keep the historical implementation
            LOG.info('Storage type is KVMV3.')
            volume_file['snapshot_id'] = snapshot_url.split('/').pop().strip()
        else:
            # Others: a openstack snapshot id
            volume_file['snapshot_id'] = snapshot_id

        # Get source volume id
        volume_file['source_volume_id'] = source_volume_id

        # Set default value to reserved parameters
        volume_file['bootable'] = self.__get_none_clone_volume_bootable(source_volume_id)
        volume_file['image_id'] = None
        volume_file['clone_volume_url'] = None

        LOG.info("The volume_file [%s] is constructed(snapshot-backup).", volume_file)
        return volume_file

    # --- End M Version Snapshot Backup Support ---

    def _get_snapshot_backup_type(self, backup):
        display_desc = backup['display_description']
        if display_desc in (None, ''):
            return -1
        try:
            json_display_desc = Utils.transfer_json_to_dict(display_desc)
            st = json_display_desc.get('ST')
        except Exception as msg:
            LOG.info("get ST from description failed, invalid json format.")
            LOG.error(msg)
            return -1
        if st in (None, ''):
            LOG.error('ST is none.')
            return -1
        if (st != SNAPSHOT_BACKUP_MACHINE and st != SNAPSHOT_BACKUP_SINGLE_VOL):
            LOG.error('ST is not machine and single volume.')
            return -1
        return st

    def get_volume_az(self, volume_id):
        vol = call_db_fun(self.db.volume_get, self.context, volume_id)
        return vol['availability_zone']

    def get_more_info_by_rep_name(self, rep_name):

        repository_uuid = None
        pool_name = None
        unit_path = None
        unit_filesystem_type = None
        pool_id = None
        unit_info = None

        repository_result = self._get_object(
            self.ebackup_manager_repository_list_uri, rep_name)
        error_code = self._get_error(repository_result)
        if error_code != 0:
            msg = self._convert_error(error_code, "get repository failed")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        rep_meta_data = self._get_data(repository_result)
        b_rep_exist = len(rep_meta_data)
        if b_rep_exist != 0:
            LOG.info('Get respository by backup!')
            rep_data = None
            for repository in rep_meta_data:
                if repository["NAME"] == rep_name:
                    rep_data = repository
                else:
                    LOG.info('responseName : %s, wantName: %s' %
                             (repository["NAME"], rep_name))
            if rep_data is None:
                msg = self._convert_error(error_code, "not found repository named %s" % rep_name)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            repository_uuid = rep_data["UUID"]
            pool_name = rep_data["PARENTNAME"]
            pool_id = rep_data["PARENTID"]

        ebk_client = EbackupConnectionClient(self.__get_connection())
        storage_pool, error_code, error_msg = ebk_client.get_storage_pool_by_name(pool_name)
        if error_code != 0 or storage_pool is None or len(storage_pool) == 0:
            msg = self._convert_error(error_code, error_msg)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        data = storage_pool[0]
        unit_info = data["BRICKLIST"]
        LOG.info(str(unit_info))
        unit_info_data = json.loads(unit_info)
        unit_info_dict = unit_info_data[0]
        unit_name = unit_info_dict['NAME']

        result = self._get_object(
            self.ebackup_manager_unit_list_uri, unit_name)
        error_code = self._get_error(result)
        if error_code != 0:
            msg = self._convert_error(error_code, "get unit failed")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        meta_data = self._get_data(result)
        if len(meta_data) != 0:
            LOG.info('get unit information!!')
            data = meta_data[0]
            unit_path = data["PATH"]
            unit_filesystem_type = data["FILESYSTEMTYPE"]

        return repository_uuid, unit_filesystem_type, unit_path, pool_id

    def update_om_hostid_for_backup(self, backup, backup_id_input, update_b_id=False):
        backup_id = backup_id_input
        if backup_id_input in (None, ''):
            backup_id = backup['id']
            if len(backup_id) != cst.LENGTH_LIMIT_36:
                msg = ('backup id length does not comply, error length:%d') % len(backup_id)
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)

        backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
        json_service_metadata = backup_in_database.get('service_metadata')
        if json_service_metadata in (None, ''):
            metadata = {}
        else:
            metadata = json.loads(json_service_metadata)
        cascade_om_hostname = Utils.get_hostname()
        if update_b_id:
            self._change_value_by_keys(metadata, OMID=cascade_om_hostname, B_ID=cascade_om_hostname)
        else:
            self._change_value_by_keys(metadata,
                                       OMID=cascade_om_hostname)
        self._update_backup(backup_id, service_metadata=json.dumps(metadata))
        LOG.info('update_om_hostid_for_backup: backup_id : %s, om_hostid: %s' %
                 (str(backup_id), str(cascade_om_hostname)))
        return cascade_om_hostname

    def update_database_for_backup(self, backup):
        if getattr(backup, cst.PLAN_NAME, None) is None:
            plan_name = self._get_plan_name(backup)
        else:
            plan_name = backup[cst.PLAN_NAME]
            LOG.info("specific plan name is %s" % plan_name)
        backup_plan_result = self._get_object(
            self.ebackup_manager_plan_list_uri, plan_name)
        error_code = self._get_error(backup_plan_result)
        if error_code != 0:
            msg = self._convert_error(error_code, "get backup plan failed.")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        plan_meta_data = self._get_data(backup_plan_result)
        plan_exist = len(plan_meta_data)

        if plan_exist != 0:
            LOG.info('backup plan is exist!')
            plan_data = plan_meta_data[0]
            plan_uuid = plan_data["UUID"]
            policy_name = plan_data["POLICYNAME"]
            repository_name = plan_data["NAMESPACENAME"]
            LOG.info("backup's relate repository is %s" % repository_name)
        else:
            msg = self._convert_error(error_code, "no backup plan.")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        repository_uuid, unit_filesystem_type, unit_path, pool_id = self.get_more_info_by_rep_name(repository_name)

        data_layout = self._get_data_layout_from_name(policy_name)
        volume_uuid = self._get_cascaded_volume_id(backup)
        backup_image_path = repository_uuid + "/" + plan_uuid + "/" + volume_uuid

        backup_id = backup['id']

        if (len(repository_uuid) == 0 or len(plan_uuid) == 0 or len(volume_uuid) == 0 or len(unit_path) == 0):
            LOG.error("get ebackup infor error")
            return 1

        backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
        metadata = json.loads(backup_in_database.get('service_metadata'))

        self._change_value_by_keys(metadata,
                                   BT=unit_filesystem_type,
                                   BP=unit_path,
                                   SP=backup_image_path,
                                   DL=data_layout)

        self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
        return 0

    def sync_vm_tag(self, vm_id, project_id, backup):
        try:
            LOG.info("context.roles: %s", self.context.roles)
            if 'op_gated_csbs_tag' in self.context.roles:
                LOG.info("vm tag function is disabled in user token")
                return
            vm_tags = NovaApi(ebackup_driver_conf.ebackup_nova_endpoint,
                              ebackup_driver_conf.ebackup_nova_ecs_api).get_vm_tags(self.context, vm_id, project_id)
            if vm_tags is not None and len(vm_tags) > 0:
                updated_tags = call_db_fun(self.db.backup_tags_private_update, self.context, backup['id'], vm_tags,
                                           False)
                LOG.info("udpate vm tag success: %s", updated_tags == vm_tags)

                ExpensesEvent.sync_tags_to_cascading(backup['display_name'], vm_tags)
            else:
                LOG.info("vm tag is empty: %s", vm_tags)
        except Exception as e:
            LOG.error("sync vm tag failed, reason is %s, ebackup driver ingore it" % e)

    def sync_vbs_tag(self, volume_id, backup):
        try:
            if (ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD and hasattr(self.db, 'volume_tags_get')
                and hasattr(self.db, 'backup_tags_private_update')):
                vbs_tags = call_db_fun(self.db.volume_tags_get, self.context, volume_id)
                if vbs_tags is not None and len(vbs_tags) > 0:
                    updated_tags = call_db_fun(self.db.backup_tags_private_update, self.context, backup['id'], vbs_tags,
                                               False)
                    LOG.info("udpate vbs tag success: %s", updated_tags == vbs_tags)
                    ExpensesEvent.sync_tags_to_cascading(backup['display_name'], vbs_tags)
                else:
                    LOG.info("vbs tag is empty: %s", vbs_tags)
        except Exception as e:
            LOG.error("sync vbs tag failed, reason is %s, ebackup driver ingore it" % e)

    @profiler.trace("ebackup driver:invoke to create backup Image")
    @HandleErrorDesc(ErrorCode.BD_BACKUP_FAILED)
    def backup(self, backup, device_file, backup_metadata=False, verify=False):
        """Backup the given volume to eBackup.

        eBackup performs a backup of a volume. The volume_file is used
        to determine the path of the block device that eBackup will back-up.

        :param backup: backup information for volume
        :param device_file: file object representing the volume in Juno,but a file descriptor of the volume in Mitaka
        :param backup_metadata: whether or not to backup volume metadata
        :raises InvalidBackup
        """
        cashier = ExpensesEvent(self.db, 'backup')
        try:
            self.error_info.error_code = ErrorCode.BD_BACKUP_FAILED
            self.__task_type = 'backup'
            self.__backup = backup
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()
            is_m_version_snapshot_backup = False
            is_m_version = False
            self.update_om_hostid_for_backup(backup, "", True)
            self.init_evs_body(backup, 'backup')
            if device_file is None:  # cinder version is M
                is_m_version = True
                LOG.info("cinder version is M.")
                # M version snapshot backup (backup has snapshot_id and it's not none)
                if ('snapshot_id' in backup.keys()) and (backup.get('snapshot_id') is not None):
                    is_m_version_snapshot_backup = True
                    LOG.info("Snapshot backup in M version.")
                    volume_file = self.__get_backup_snapshot_volume_file(backup)
                # M version link clone volume backup
                else:
                    LOG.info("Link clone volume backup in M version.")
                    volume_file = self.__get_backup_volume_file(backup)
                self.set_fusion_storage_ip_for_connection(volume_file, cst.TASKTYPE_BACKUP)
            else:  # cinder version is J
                LOG.info("cinder version is J.")
                volume_file = device_file
                # Get AZ name
                self.__availability_zone = self.get_volume_az(backup['volume_id'])

            tenant_id_legal = self._check_length(backup.get('project_id', ''), cst.LENGTH_LIMIT_255)
            volume_id_legal = self._check_length(volume_file.get('source_volume_id', ''), cst.LENGTH_LIMIT_36)
            backup_id_legal = self._check_length(backup.get('id', ''), cst.LENGTH_LIMIT_36)
            if tenant_id_legal or volume_id_legal or backup_id_legal:
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, 'some parameter length does not comply')
                LOG.error(error_msg)
                LOG.error('parameter error! tenant: %s, volume: %s, backup %s' % (
                    backup.get('project_id', ''), volume_file.get('source_volume_id', ''), backup.get('id', '')))
                raise Exception(error_msg)
            else:
                LOG.info('Begin hwsebackup backup,tenant: %s, volume: %s, backup %s' % (
                    backup.get('project_id', ''), volume_file.get('source_volume_id', ''), backup.get('id', '')))

            if volume_file.get('machine_id', None):
                self.sync_vm_tag(volume_file.get('machine_id'), backup.get('project_id'), backup)
            elif self._get_snapshot_backup_type(backup) in (SNAPSHOT_BACKUP_SINGLE_VOL, -1):
                self.sync_vbs_tag(volume_file.get('source_volume_id'), backup)
            else:
                LOG.info('Do not sync tags')

            backup_in_database = call_db_fun(self.db.backup_get, self.context, backup.get('id'))
            metadata = Utils.transfer_json_to_dict(backup_in_database.get('service_metadata'))

            # M version snapshot backup
            if is_m_version_snapshot_backup:
                self.__get_backup_volume_type(volume_file.get('source_volume_id'), volume_file)
                st = self._get_snapshot_backup_type(backup)
                if st == -1:
                    st = 0
                if st == 0:
                    LOG.info('its a single volume backup.')
                self._change_value_by_keys(metadata,
                                           Type=volume_file['backup_type'],
                                           progress='1',
                                           bootable=volume_file['bootable'],
                                           ST=st,
                                           VT=volume_file['volume_type_name'])
                if ebackup_driver_conf.ebackup_vm_sdr:
                    self._change_value_by_keys(metadata, VMID=volume_file.get('machine_id', ''))
            # J version and m version link clone backup
            else:
                self._change_value_by_keys(metadata,
                                           Type=volume_file['backup_type'],
                                           progress='1',
                                           bootable=volume_file['bootable'],
                                           snap_id=volume_file['snapshot_id'],
                                           ST=0)
                backup['volume_id'] = volume_file.get('source_volume_id')

            if is_m_version and is_infiniBand_env and self.__fusionstorage_ip:
                pht_type = 1 if self.__get_connection().get_phy_network_type() == 'ib' else 0
                self._change_value_by_keys(metadata, IB=pht_type)

            self._update_backup(backup.get('id'),
                                service_metadata=json.dumps(metadata),
                                volume_id=volume_file.get('source_volume_id'))

            if self._get_cloud_service_type(backup) == 'CBR':
                uranus = Uranus(self.context, self.db,
                                call_db_fun(self.db.backup_get, self.context, backup.get('id')),
                                self.__get_connection(), ebackup_driver_conf, volume_file, self.evs_body, verify)
                uranus.backup_engine()
            else:
                self._check_storage_ralate(backup)
                if is_m_version:
                    self.update_database_for_backup(backup)
                self._run_backup_plan(backup, volume_file)
                LOG.info(volume_file.get('source_volume_id') + ' hwsebackup backup end.')
                backup = call_db_fun(self.db.backup_get, self.context, backup.get('id'))
                cashier.send_expenses_event(self.context, backup, backup, successd=True)

        except Exception as msg:
            LOG.error(msg)
            LOG.error('Origin stack is:%s' % traceback.format_exc())
            backup = call_db_fun(self.db.backup_get, self.context, backup['id'])
            cashier.send_expenses_event(self.context, backup, backup, successd=False)
            raise exception.InvalidBackup(msg)

    def _check_backuping_by_restore(self, uri, restore_body):
        """ check target volume status, if its backuping, waiting for 24h """
        loop_num = 0

        while True:
            if loop_num >= cst.QUERY_ONE_DAY:
                error_msg = convert_error(ErrorCode.VM_HAS_USED,
                                          'restore failed, because the target volume is backuping and exceed 24 hours.')
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            put_result = self.__get_connection().put(uri, restore_body)
            error_number = self._get_error(put_result)
            if error_number == ErrorCode.VM_HAS_USED:
                LOG.info('The target volume is backuping, please waite it complete.')
                time.sleep(cst.TIME_SLEEP_30)
                loop_num += 1
                continue
            elif error_number != 0:
                error_desc = self._get_error_description(put_result)
                self.error_info.set_error_info(error_number, error_desc)
                msg = self._convert_error(error_number, error_desc)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            else:
                put_data = self._get_data(put_result)
                request_id = put_data.get("REQUEST_ID")
                return request_id

    def _get_image_by_task(self, task_id):
        """ get image id by eBackup task id.
        """
        while True:
            task_status, task_progress = self._get_task_progress(task_id)
            LOG.debug('get image id by eBackup task id, progress is:%s' % task_progress)
            # backup failed
            if task_status == cst.TASK_STATUS_FAILED:
                return None
            if task_status == ErrorCode.TSKMG_TASK_ID_NOT_EXIST:
                return ErrorCode.TSKMG_TASK_ID_NOT_EXIST
            # backup completed
            if task_status == cst.TASK_STATUS_COMPLETED:
                backupurl = self._get_new_snap_uuid(task_id)
                return backupurl

    def _find_restore_task(self, backup, metadata):
        backup_id = backup['id']
        task_id_metadata = metadata.get('ebk_T_I')
        if task_id_metadata == '':
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "ebk_T_I in service_metadata is empty.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        LOG.info('Begin hwsebackup restore, ebk_T_I :%s' % task_id_metadata)

        task_id = ''
        request_id = ''
        if task_id_metadata.startswith('R_'):
            request_id = task_id_metadata[2:]
        else:
            task_id = task_id_metadata

        task_type_restore = cst.TASKTYPE_VOL_RESTORE
        if metadata.get("C-Region"):
            task_type_restore = cst.TASKTYPE_VOL_REMOTE_RESTORE

        if task_id in (None, ''):
            task_id = self._get_task_id(task_type_restore, request_id)
            self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
            self._update_backup(backup_id, service_metadata=json.dumps(metadata))

        while True:
            time.sleep(cst.TIME_SLEEP_30)
            task_infos = self.get_task_info(task_id)

            if not isinstance(task_infos, dict):
                LOG.error("task : %s failure,failed to get task info" % task_id)
                break

            task_status, task_progress, task_details = self._get_task_progress_and_details(task_infos,
                                                                                           task_type_restore)
            LOG.info('restore task progress is:%s' % task_progress)
            if task_status == cst.TASK_STATUS_FAILED:
                msg = ("eBackup Restore VM disk job failed, job id is:" + str(task_id))
                fail_reason_str = self._get_task_detail(task_details, msg)
                LOG.error("Restore job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                raise exception.InvalidBackup(fail_reason_str)

            # update task progress.
            if task_progress == '0':
                task_progress = '1'
            self._change_value_by_keys(metadata, progress=str(task_progress))
            self._update_backup(backup_id, service_metadata=json.dumps(metadata))

            if task_status == cst.TASK_STATUS_COMPLETED:
                self._change_value_by_keys(metadata, progress='0', ebk_T_I='')
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))
                break

    # ---begin add 2016-11-1----
    def __get_volume_storage_type_from_provider_location(self, volume, is_rdm):
        storageType = ''
        provider_location_str = volume.get("provider_location")
        if provider_location_str is None:
            LOG.error('The provider location does not exist.')
        else:
            provider_location_list = provider_location_str.split(',')
            for key in provider_location_list:
                if key.startswith('storageType='):
                    storageType_list = key.split('=')
                    if len(storageType_list) == 2:
                        storageType = storageType_list[1]
                    break
        LOG.info("the storage type in provider location is: %s." % str(storageType))
        if "DSWARE" == storageType:
            if is_rdm:
                volume_storage_type = cst.STORAGE_TYPE_VRM_FUSION_RDM
            else:
                volume_storage_type = cst.STORAGE_TYPE_VRM_FUSION_STORAGE
        elif "" == storageType:
            LOG.info('The storage type in provider location does not exist, set default is KVM+FustionStorage.')
            volume_storage_type = cst.STORAGE_TYPE_KVM_FUSION_STORAGE
        else:
            LOG.error('Invalid storage type: %s' % storageType)
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "Invalid storage type")
            raise exception.InvalidBackup(error_msg)
        return volume_storage_type

    def __get_volume_storage_type_from_provider_location_cinderup(self, volume):
        provider_location_str = volume.get("provider_location")
        if provider_location_str is None:
            LOG.error('The provider location does not exist.')
            return -1

        try:
            provider_location = json.loads(provider_location_str)
            storageType = provider_location.get('storage_type', '')
            storage_type_dict = {
                "FC_DSWARE": cst.STORAGE_TYPE_VRM_FUSION_STORAGE,
                "FusionStorage": cst.STORAGE_TYPE_KVM_FUSION_STORAGE
            }
            LOG.info("storage type in provider location is: %s." % str(storageType))

            return storage_type_dict.get(storageType, -1)

        except:
            LOG.info("provider_location is not json")

        return -1

    def __get_volume_offset_size_from_provider_location(self, volume):
        volume_offset_str = ''
        provider_location_str = volume.get("provider_location")
        if provider_location_str is None:
            LOG.error('The get_volume_offset_size provider location does not exist.')
        else:
            try:
                provider_location = json.loads(provider_location_str)
            except:
                LOG.info("get_volume_offset_size provider_location is not json")
                return volume_offset_str
            volume_offset = provider_location.get('offset')
            if volume_offset not in (None, ''):
                volume_offset_str = str(volume_offset)
        return volume_offset_str

    def __get_volume_storage_type(self, volume):
        volume_storage_type = self.__get_volume_storage_type_from_provider_location_cinderup(volume)
        if -1 != volume_storage_type:
            return volume_storage_type

        # Check volume_type_extra_specs for RDM
        is_rdm = False
        volume_storage_type_str = ''
        volume_metadata = volume.get("volume_metadata")
        for metadata in volume_metadata:
            if metadata.key == 'StorageType':
                volume_storage_type_str = metadata.value
            if metadata.key == 'hw:passthrough':
                is_rdm = (metadata.value == "true" or metadata.value == "True")

        LOG.info("the storage type string of volume is: %s." % str(volume_storage_type_str))
        if not is_rdm:
            volume_type_extra_specs = volume.get("volume_type_extra_specs")
            if volume_type_extra_specs is not None:
                for type_extra_specs in volume_type_extra_specs:
                    if type_extra_specs.key == 'hw:passthrough':
                        is_rdm = type_extra_specs.value
                        break
        if volume_storage_type_str in ['OceanStorV3', 'OceanStoreV3', 'DoradoV3', 'OceanStorV5', 'DoradoV6',
                                       'OceanStorV6']:
            volume_storage_type = cst.STORAGE_TYPE_KVMV3
        elif "FusionStorage" == volume_storage_type_str:
            volume_storage_type = cst.STORAGE_TYPE_KVM_FUSION_STORAGE
        elif "FC_DSWARE" == volume_storage_type_str:
            if is_rdm:
                volume_storage_type = cst.STORAGE_TYPE_VRM_FUSION_RDM
            else:
                volume_storage_type = cst.STORAGE_TYPE_VRM_FUSION_STORAGE
        elif "FC_advanceSan" == volume_storage_type_str:
            if is_rdm:
                volume_storage_type = cst.STORAGE_TYPE_VRMV3RDM
            else:
                volume_storage_type = cst.STORAGE_TYPE_VRMV3
        elif "" == volume_storage_type_str:
            LOG.info('The storage type in volume metadata does not exist.')
            volume_storage_type = self.__get_volume_storage_type_from_provider_location(volume, is_rdm)
        else:
            LOG.error('Invalid storage type: %s' % volume_storage_type_str)
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "Invalid storage type")
            raise exception.InvalidBackup(error_msg)
        LOG.info("the storage type is: %s." % volume_storage_type)
        return volume_storage_type

    def __get_kvm_fusionstorage_manager_ip(self, volume):
        volume_metadata = volume.get('volume_metadata')
        if volume_metadata is None:
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR,
                                      "DSWARE get manager ip failed,volume metadata is null!")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        for metadata in volume_metadata:
            if metadata.key.lower() == 'manager_ip':
                manager_ip = metadata.value.lower()
                if manager_ip in (None, ''):
                    error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "manager ip is none or empty.")
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)
                return manager_ip

        error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "DSWARE get manager ip failed!")
        LOG.error(error_msg)
        raise exception.InvalidBackup(error_msg)

    def __get_kvm_fusionstorage_pool_id(self, host):
        # host format: 'hostid@backend#poolid'
        # other formats: return 'default', and the pool id would be zero.
        if host:
            if len(host.split('#', 1)) == 2:
                pool_id = host.split('#')[1]
                if pool_id in (None, ''):
                    error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "pool id is none or empty.")
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)
                return pool_id
        return '0'

    def __get_kvm_fusionstorage_manager_ip_and_pool_id(self, volume):
        manager_ip = self.__get_kvm_fusionstorage_manager_ip(volume)
        host = volume.get('host')
        pool_id = self.__get_kvm_fusionstorage_pool_id(host)
        return manager_ip, pool_id

    def __get_ip_and_pool_from_provider(self, provider):
        # provider must be volume or source_snapshot
        try:
            provider_location = json.loads(provider.get("provider_location"))
            return provider_location.get('ip', ''), provider_location.get('pool', '')
        except Exception as error:
            LOG.error('The provider location is illegal %s.', error)
        return ('', '')

    def __get_vol_name_from_provider_location(self, volume):
        try:
            provider_location = json.loads(volume.get("provider_location"))
            return provider_location.get('vol_name')
        except Exception as error:
            LOG.error('The get_vol_name provider location is illegal %s.', error)

        return ''

    def __get_snap_name_from_provider_location(self, source_snapshot):
        try:
            provider_location = json.loads(source_snapshot.get("provider_location"))
            return provider_location.get('snap_name')
        except Exception as error:
            LOG.error('The get_vol_name provider location is illegal %s.', error)

        return ''

    def __get_vrm_fusionstorage_manager_ip_and_pool_id(self, volume):
        volume_metadata = volume.get('volume_metadata')
        if volume_metadata is None:
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the volume metadata is null!")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        volInfoUrl = ''
        for metadata in volume_metadata:
            if metadata.key.lower() == 'volinfourl':
                volInfoUrl = metadata.value
                if volInfoUrl in (None, ''):
                    error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "volInfoUrl is none or empty.")
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)
        fusionstorage_info = volInfoUrl.split("/")
        if len(fusionstorage_info) < 4:
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "get manage ip or pool id failed.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        manager_ip = fusionstorage_info[2]
        pool_id = fusionstorage_info[3]
        if manager_ip in (None, '') or pool_id in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the manage ip or pool id is none or empty.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)

        return manager_ip, pool_id

    def __get_v3_sn_and_lunid_from_snapshots(self, source_snapshot):
        provider_auth_str = source_snapshot.get('provider_auth')
        if provider_auth_str in (None, ''):
            LOG.error("provider_auth_str is None")
            return '', ''
        provider_auth = json.loads(provider_auth_str)
        sn = provider_auth.get('sn')
        if sn in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the sn in provider_auth is None.")
            LOG.error(error_msg)
        lun_id = provider_auth.get('lun_id')
        if lun_id in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the lun_id in provider_auth is None.")
            LOG.error(error_msg)
        return sn, lun_id

    def __get_v3_sn_and_lunid_from_volume(self, volume):
        LOG.info("enter __get_v3_sn_and_lunid_from_volume")
        replication_driver_data_str = volume.get('replication_driver_data')
        if replication_driver_data_str in (None, ''):
            LOG.error("replication_driver_data_str is None")
            return '', ''
        replication_driver_data = json.loads(replication_driver_data_str)
        sn = replication_driver_data.get('sn')
        if sn in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the sn in replication_driver_data is None.")
            LOG.error(error_msg)
        lun_id = replication_driver_data.get('lun_id')
        if lun_id in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the lun_id in replication_driver_data is None.")
            LOG.error(error_msg)
        return sn, lun_id

    def __get_v3_sn(self, volume):
        replication_driver_data_str = volume.get('replication_driver_data')
        if replication_driver_data_str in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the replication_driver_data is None.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        replication_driver_data = json.loads(replication_driver_data_str)
        storage_sn = replication_driver_data.get('sn')
        if storage_sn in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the storage sn is None.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        return storage_sn

    def __get_fusionstorage_volume_url(self, volume, storage_type):
        manager_ip, pool_id = self.__get_ip_and_pool_from_provider(volume)
        volume_name = self.__get_vol_name_from_provider_location(volume)
        if ((manager_ip not in (None, '')) and (pool_id not in (None, '')) and (volume_name not in (None, ''))):
            volume_url = 'http://' + manager_ip + '/' + str(pool_id) + '/' + volume_name
            return volume_url

        if cst.STORAGE_TYPE_VRM_FUSION_STORAGE == storage_type or cst.STORAGE_TYPE_VRM_FUSION_RDM == storage_type:
            volume_metadata = volume.get('volume_metadata')
            if volume_metadata is None:
                error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the volume metadata is null!")
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            volInfoUrl = ''
            for metadata in volume_metadata:
                if metadata.key.lower() == 'volinfourl':
                    volInfoUrl = metadata.value
                    if volInfoUrl in (None, ''):
                        error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "volInfoUrl is none or empty.")
                        LOG.error(error_msg)
                        raise exception.InvalidBackup(error_msg)
            return volInfoUrl
        else:
            manager_ip, pool_id = self.__get_kvm_fusionstorage_manager_ip_and_pool_id(volume)
        volume_name = volume.get('name')
        if volume_name in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the volume name is None.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        volume_url = 'http://' + manager_ip + '/' + str(pool_id) + '/' + volume_name
        return volume_url

    def __get_v3_volume_url(self, volume):
        # volume url format like: https://storageSN/lunid
        # get sn and lunId from replication_driver_data field prior, then enter original branch if not exist.
        storage_sn, lun_id = self.__get_v3_sn_and_lunid_from_volume(volume)
        if storage_sn in (None, ''):
            storage_sn = self.__get_v3_sn(volume)
        if lun_id in (None, ''):
            lun_id = volume.get("provider_location")
        if lun_id in (None, ''):
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "the lun id is None.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        volume_url = 'https://' + storage_sn + '/' + lun_id

        return volume_url

    def __get_last_backup(self, backup_list, creating_status=False, currentid="", backup_type=""):
        last_backup = None
        if backup_type == "backup":
            image_types = [0, 1]
        else:
            image_types = [4, 5]
        if backup_list:
            for back_tmp in backup_list:
                if back_tmp['status'] == cst.BACKUP_STATUS_CREATING \
                        and creating_status is True and back_tmp['id'] != currentid:
                    if not back_tmp['service_metadata']:
                        continue
                    last_metadata = json.loads(back_tmp.get('service_metadata'))
                    last_backup_storage_type = last_metadata.get('BT')
                    last_backup_datalayout = last_metadata.get('DL')
                    if not last_backup_storage_type or not last_backup_datalayout:
                        continue
                    LOG.info("Get Creating backup %s" % back_tmp['id'])
                elif (back_tmp['status'] != cst.BACKUP_STATUS_AVALIBLE  # "available"
                      and back_tmp['status'] != cst.BACKUP_STATUS_RESTORING):  # "restoring"
                    continue
                last_metadata = json.loads(back_tmp.get('service_metadata'))
                last_backup_type = last_metadata.get('ST')
                if last_backup_type in image_types:
                    if not last_backup:
                        last_backup = back_tmp
                    if last_backup['created_at'] < back_tmp['created_at']:
                        last_backup = back_tmp
        return last_backup

    def __get_restore_volume_file(self, backup, volume_id):
        volume_file = {}

        # 1. set restore_type
        LOG.info("start set restore type")
        source_volume_id = backup['volume_id']
        volume_file['restore_type'] = 0
        if source_volume_id == volume_id:
            volume_file['restore_type'] = 1

        # 2. set storage type
        LOG.info("start set storage type")
        volume_info = call_db_fun(self.db.volume_get, self.context, volume_id)  # get the target volume info
        volume_storage_type = self.__get_volume_storage_type(volume_info)
        volume_file['storage_type'] = int(volume_storage_type)

        # Get AZ name
        self.__availability_zone = volume_info['availability_zone']

        # 3. set volume url
        LOG.info("start set volume url")
        if cst.STORAGE_TYPE_VRM_FUSION_STORAGE == volume_storage_type \
                or cst.STORAGE_TYPE_KVM_FUSION_STORAGE == volume_storage_type \
                or cst.STORAGE_TYPE_VRM_FUSION_RDM == volume_storage_type:
            volume_file['volume_url'] = self.__get_fusionstorage_volume_url(volume_info, volume_storage_type)
        else:
            volume_file['volume_url'] = self.__get_v3_volume_url(volume_info)

        # check. The volume which migrated from xen pod.Can't restore operation system disk.
        if volume_info['bootable'] and volume_storage_type in (cst.STORAGE_TYPE_KVMV3,
                                                               cst.STORAGE_TYPE_KVM_FUSION_STORAGE):
            service_md = json.loads(backup['service_metadata'])
            if service_md.get('migrate', '') == cst.XEN_TO_KVM:
                error_msg = convert_error(ErrorCode.BD_BACKUP_STATE_NOT_RIGHT,
                                          "The backup which migrate from Xen POD cant restore to KVM system disk!")
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)

        self.set_fusion_storage_ip_for_connection(volume_file, cst.TASKTYPE_VOL_RESTORE)
        # 4. set clone volume url
        LOG.info("start set clone volume url")
        volume_file['clone_volume_url'] = None

        LOG.info("The restore volume_file [%s] is constructed", volume_file)
        return volume_file

    # ---end add 2016-11-1----

    def get_import_type(self, backup):
        import_type = 0
        if backup['service_metadata'] not in ("", None):
            metadata = Utils.transfer_json_to_dict(backup.get('service_metadata'))
            if (metadata not in ("", None)) and "IT" in metadata:
                import_type = metadata['IT']
        return import_type

    def is_remote_replication_error_backup(self, backup):
        flag = False
        metadata = Utils.transfer_json_to_dict(backup.get('service_metadata'))
        if cst.IMPORT_TYPE_REMOTE_REPLICATION == metadata.get('IT') and cst.BACKUP_STATUS_ERROR == metadata.get('TS'):
            flag = True
        return flag

    def _check_encrypted_metadata_valid(self, is_encrypted, volume_cmkId):
        if is_encrypted is not None and str(is_encrypted) == '1':
            if volume_cmkId is None:
                return False
        return True

    def do_lld_snapshot_rollback(self, backup, volume_id, pool_id):
        try:
            snapshot_id = backup['snapshot_id']
            # get snapshot for last backup, raise exception if it dose not exist
            last_snapshot = call_db_fun(self.db.snapshot_get, self.context, snapshot_id)
            snapshot_name = self.__get_snap_name_from_provider_location(last_snapshot)
            if snapshot_name in (None, ''):
                LOG.error("snapshot %s of last backup %s dose not exist." % (snapshot_id, backup['id']))
                return False
            target_volume = call_db_fun(self.db.volume_get, self.context, volume_id)
            volume_name = self.__get_vol_name_from_provider_location(target_volume)
            if volume_name in (None, ''):
                LOG.error("get volume name of volume %s failed." % volume_id)
                return False

            LOG.info("rollback volume %s with snapshot %s." % (volume_name, snapshot_name))
            conn = self.__get_connection()
            if conn is None:
                return False

            dsware_client = FSPythonApi(conn.get_fusion_storage_ip(), conn.get_fusion_storage_agent_ip())
            exe_result = dsware_client.snapshot_rollback(snapshot_name,
                                                         volume_name, pool_id)
            LOG.info("snapshot rollback finished, result code is %s", exe_result)
            if exe_result != 0:
                return False
            else:
                return True
        except Exception as msg:
            LOG.info(msg)
            return False

    def do_lld_restore(self, backup, volume_id):
        LOG.info("do lld restore start, backup id:%s, volume_id:%s", backup['id'], volume_id)
        if self.not_do_lld_in_redo_restore:
            return
        conn = self.__get_connection()
        if conn is None:
            return False

        # get current backup datalaylout
        json_meta_data = backup['service_metadata']
        backup_meta_data = json.loads(json_meta_data)
        backup_data_layout = backup_meta_data.get("DL")
        backup_native_lld = backup_meta_data.get("LLD", 0)
        hybird_image = backup_meta_data.get("HI", 0)
        cbr_backup = backup_meta_data.get("CBR", 0)
        # If the condition is not satisfied then return
        LOG.info("context.roles: %s", self.context.roles)

        if ebackup_driver_conf.ebackup_lazyloading == '0' or conn.get_fusion_storage_agent_ip() == '' or \
                        conn.get_fusion_storage_ip() == 'default' or ebackup_driver_conf.ebackup_env_type != ENV_PUBLIC_CLOUD:
            LOG.info("lld ebackup_lazyloading config is unsuitable")
            return False
        elif ebackup_driver_conf.ebackup_lazyloading == '1':
            if ('op_gated_bklld' in self.context.roles) or hybird_image or cbr_backup or (
                            str(backup_data_layout) != '0' and backup_native_lld != 1):
                LOG.info("lld is 1 but image or tenant role is unsuitable")
                return False
        elif ebackup_driver_conf.ebackup_lazyloading == '2':
            if hybird_image or cbr_backup or (str(backup_data_layout) != '0' and backup_native_lld != 1):
                LOG.info("lld is 2 but image is unsuitable")
                return False
        elif ebackup_driver_conf.ebackup_lazyloading == '3':
            if ('op_gated_bklld' in self.context.roles) or (str(backup_data_layout) != '0' and backup_native_lld != 1):
                LOG.info("lld is 3 but image or tenant role is unsuitable")
                return False
        elif ebackup_driver_conf.ebackup_lazyloading == '4':
            if str(backup_data_layout) != '0' and backup_native_lld != 1:
                LOG.info("lld is 4 but image is unsuitable")
                return False
        else:
            LOG.info("unknown default.lazyloading value!")
            return False

        volume = objects.Volume.get_by_id(self.context, volume_id)
        backup_in_database = call_db_fun(self.db.backup_get, self.context, backup['id'])
        volume_info = call_db_fun(self.db.volume_get, self.context, volume_id)
        try:
            # update ebk_T_I with lld flag
            volume_storage_type = self.__get_volume_storage_type(volume_info)
            volume_url = self.__get_fusionstorage_volume_url(
                volume_info, volume_storage_type)
            pool_id = volume_url.split("/")[3]
            json_service_metadata = backup_in_database.get('service_metadata')
            update_metadata = json.loads(json_service_metadata)
            self._change_value_by_keys(update_metadata, progress=str('1'), ebk_T_I=cst.TAG_LLD_START)
            self._update_backup(backup['id'], service_metadata=json.dumps(update_metadata))

            rollbackup_result = self.do_lld_snapshot_rollback(backup,
                                                              volume_id,
                                                              pool_id)
            if rollbackup_result:
                self._change_value_by_keys(update_metadata, ebk_T_I=cst.TAG_LLD_SUCCESS)
                self._change_value_by_keys(update_metadata, progress=str('0'))
                self._update_backup(backup['id'], service_metadata=json.dumps(update_metadata))
                LOG.info("snapshot rollback succ")
                return True
            else:
                LOG.info("execute original lld restore flow since snapshot rollback failed")

            # get params for lld retore
            meta_data = volume.get('metadata')
            dsware_volume_name = volume_url.split("/")[4]
            volume_size = volume['size'] * 1024
            isThin = 0
            cacheFlag = 0
            replace = 1
            hw_passthrough = meta_data.get('hw:passthrough', None)

            volume_type_unnecessary_check_offset = [cst.STORAGE_TYPE_KVM_FUSION_STORAGE, cst.STORAGE_TYPE_KVMV3]
            if volume_storage_type not in volume_type_unnecessary_check_offset:
                if hw_passthrough and str(hw_passthrough).lower() == 'true':
                    offset = 0
                else:
                    offset = 4096
                    volume_size += 1
            else:
                offset = 0

            is_encrypted = meta_data.get('__system__encrypted', None)
            cmkId = volume['metadata'].get('__system__cmkid', None)
            if not self._check_encrypted_metadata_valid(is_encrypted, cmkId):
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG,
                                          "__system__encrypted is set but __system__cmkid is not set")
                raise exception.InvalidBackup(error_msg)
            volume_auth_token = self.context.auth_token

            image_size = backup['size'] * 1024  # GB to MB

            http_method = "https" if ebackup_driver_conf.ebackup_storage_s3_protocol == '0' else "http"

            bp = backup_meta_data.get('BP')
            ip = bp.split(":", 1)[0]
            port = 0
            if len(bp.split(":")) == 3:
                port = bp.split(":")[1]
            bucket_name = bp.split("/", 1)[-1]

            object_name = backup_meta_data.get('SP')
            accessKey = Utils.decrypt_password(ebackup_driver_conf.ebackup_storage_username)
            secretKey = Utils.decrypt_password(ebackup_driver_conf.ebackup_storage_password)

            is_backup_encrypted = '0'
            backup_vk = backup_meta_data.get("VK")
            backup_cmkId = backup_meta_data.get("CMKID")
            if backup_vk and backup_cmkId:
                is_backup_encrypted = '1'

            source_volume_id = backup['volume_id']

            backupUrl = backup_meta_data.get("backupurl")

            dataSrcUrl = {'source': 'uds',
                          'format': 'ebk',
                          'proto': http_method,
                          'ip': str(ip),
                          'port': str(port),
                          'bucket': str(bucket_name),
                          'object': str(object_name),
                          'ak': str(accessKey),
                          'sk': str(secretKey),
                          'encrypt': is_backup_encrypted,
                          'vk': str(backup_vk),
                          'cmkid': str(backup_cmkId),
                          'dl': str(backup_data_layout),
                          'dskid': source_volume_id,
                          'bkid': str(backupUrl)}
            json_dataSrcUrl = json.dumps(dataSrcUrl, separators=(',', ':'))
            image_id_string = str(backupUrl) + '_' + str(bp) + '/' + str(object_name) + '/_'

        except Exception as msg:
            LOG.info(msg)
            LOG.info("lld restore failed in step:%s", cst.TAG_LLD_START)
            return False

        cmd_info_dataSrcUrl = copy.deepcopy(dataSrcUrl)
        cmd_info_dataSrcUrl['ak'] = '***'
        cmd_info_dataSrcUrl['sk'] = '***'
        cmd_info_dataSrcUrl['vk'] = '***'
        cmd_info_dataSrcUrl['cmkid'] = '***'
        cmd_info = {"dsware_volume_name": dsware_volume_name,
                    "pool_id": pool_id,
                    "volume_size": volume_size,
                    "isThin": isThin,
                    "image_id_string": image_id_string,
                    "dataSrcUrl": cmd_info_dataSrcUrl,
                    "image_size": image_size,
                    "offset": offset,
                    "cacheFlag": cacheFlag,
                    "is_encrypted": is_encrypted,
                    "replace": replace}
        LOG.info("lld restore body: %s", cmd_info)

        try:
            result = True
            dsware_client = FSPythonApi(conn.get_fusion_storage_ip(), conn.get_fusion_storage_agent_ip())
            exe_result = dsware_client.create_LLD_volume(dsware_volume_name, pool_id, volume_size,
                                                         isThin, image_id_string, json_dataSrcUrl,
                                                         image_size, offset, cacheFlag,
                                                         is_encrypted, cmkId, volume_auth_token,
                                                         replace)

            if exe_result != 0:
                LOG.info("lld exe_result code:%s", exe_result)
                result = False

            # update
            if result:
                self._change_value_by_keys(update_metadata, ebk_T_I=cst.TAG_LLD_SUCCESS)
            else:
                self._change_value_by_keys(update_metadata, ebk_T_I=cst.TAG_LLD_FAILED)

            # update progress to 0 prevent query task status in ebk_server restore flow
            self._change_value_by_keys(update_metadata, progress=str('0'))
            self._update_backup(backup['id'], service_metadata=json.dumps(update_metadata))
            return result
        except exception.InvalidBackup:
            with excutils.save_and_reraise_exception():
                LOG.info("lld create done, but db update failed")
        except Exception as msg:
            LOG.info(msg)
            self._change_value_by_keys(update_metadata, progress=str('0'), ebk_T_I="R_lld_exit")
            self._update_backup(backup['id'], service_metadata=json.dumps(update_metadata))
            LOG.info("enter original restore flow since something unexpected happened")
            return False

    @profiler.trace("ebackup driver:invoke to restore volume with backupImage")
    @HandleErrorDesc(ErrorCode.BD_RESTORE_FAILED)
    def restore(self, backup, volume_id, device_file):
        """Restore the given volume backup from eBackup server.

        :param backup: backup information for volume
        :param volume_id: volume id
        :param device_file: file object representing the volume in Juno,but a file descriptor of the volume in Mitaka
        :raises InvalidBackup
        """

        metadata = None
        try:
            self.error_info.error_code = ErrorCode.BD_RESTORE_FAILED
            self.__task_type = 'restore'
            self.__backup = backup
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()
            backup_id = backup['id']
            is_m_version = False
            self.update_om_hostid_for_backup(backup, "")
            if device_file is None:  # cinder version is M
                is_m_version = True
                LOG.info("cinder version is M.")
                volume_file = self.__get_restore_volume_file(backup, volume_id)
            else:  # cinder version is J
                LOG.info("cinder version is J.")
                volume_file = device_file
                # Get AZ name
                self.__availability_zone = self.get_volume_az(volume_id)

            if is_m_version:
                is_done = self.do_lld_restore(backup, volume_id)
                if is_done:
                    backup_in_database = call_db_fun(self.db.backup_get, self.context, backup['id'])
                    json_service_metadata = backup_in_database.get('service_metadata')
                    update_metadata = Utils.transfer_json_to_dict(json_service_metadata)
                    # update task id to empty prevent for not do lld in verify
                    self._change_value_by_keys(update_metadata, ebk_T_I="")
                    self._update_backup(backup['id'], service_metadata=Utils.transfer_to_json_str(update_metadata))
                    LOG.info("%s:end lld restore end", backup["volume_id"])
                    return
                else:
                    LOG.info("enter original restore flow since lld is disable or failed")
            self.init_evs_body(backup, 'restore')
            tenant_id = backup['project_id']
            flag = self._check_length(tenant_id, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ('tenant id length does not comply, error length:%d' % len(tenant_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)

            LOG.info('Begin hwsebackup restore, tenant id:%s' % tenant_id)

            source_volume_id = self._get_cascaded_volume_id(backup)
            flag = self._check_length(source_volume_id, cst.LENGTH_LIMIT_36)
            if flag:
                msg = ('source volume id length does not comply, error length:%d' % len(source_volume_id))
                error_msg = convert_error(ErrorCode.BD_VOLUME_ID_INVALID, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)

            LOG.info('Begin hwsebackup restore, source volume id:%s' % source_volume_id)
            size = backup['size']
            # volume size length must be larger than 0 and less than 12
            if (size is None) or (len(str(size)) > cst.VOLUME_SIZE_LENGTH):
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, 'volume size is wrong')
                LOG.error('volume size is wrong, size:%s' % size)
                raise Exception(error_msg)
            volume_size = size * cst.Gi
            if len(backup_id) != cst.LENGTH_LIMIT_36:
                msg = ('backup id length does not comply, error length:%d' % len(backup_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)

            LOG.info('Begin hwsebackup restore, backup id:%s' % backup_id)

            flag = self._check_length(volume_id, cst.LENGTH_LIMIT_36)
            if flag:
                msg = ('target volume id length does not comply, error length%d' % len(volume_id))
                error_msg = convert_error(ErrorCode.BD_VOLUME_ID_INVALID, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)

            LOG.info('Begin hwsebackup restore, target volume id:%s' % volume_id)

            # update task status
            backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_service_metadata = backup_in_database.get('service_metadata')
            if json_service_metadata in (None, ''):
                msg = 'service metadata is null, restore can not begin.'
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)

            storage_type = volume_file['storage_type']
            if str(storage_type) not in (cst.STORAGE_TYPE_VRM_FUSION_STORAGE,
                                         cst.STORAGE_TYPE_KVM_FUSION_STORAGE,
                                         cst.STORAGE_TYPE_VRMVIMS,
                                         cst.STORAGE_TYPE_KVMV3,
                                         cst.STORAGE_TYPE_VRMV3,
                                         cst.STORAGE_TYPE_VRMV3RDM,
                                         cst.STORAGE_TYPE_VRM_FUSION_RDM):
                error_msg = convert_error(ErrorCode.BD_STORAGE_TYPE_NOT_SUPPORT, "storage type is error")
                LOG.error('storage type is error, error type:%s' % storage_type)
                raise Exception(error_msg)

            metadata = Utils.transfer_json_to_dict(json_service_metadata)
            progress = metadata.get('progress')
            if progress in (None, ''):
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, 'task progress is empty.')
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            elif progress == '0':
                self._change_value_by_keys(metadata, progress='1', ebk_T_I=cst.NOID)
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
            else:
                try:
                    self._find_restore_task(backup, metadata)
                    LOG.info(str(source_volume_id) + ':End hwsebackup restore.')
                    return
                except Exception as msg:
                    if str(msg) == str(cst.VERIFY_TASK_NOT_FOUND):
                        LOG.error("cannot find a restoring task.")
                    else:
                        raise exception.InvalidBackup(msg)

            region_name = ""
            image_id = metadata.get('backupurl')
            if image_id in (None, ''):
                error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, "image id is None")
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)

            LOG.info('the backup image id is:%s' % image_id)
            image = self._get_image_by_id(image_id)
            is_cross_az_restore = False
            if is_m_version:
                if not image or image.get('VALIDITY') == '3':
                    # When the validity is 3, the backup has been migrated
                    # and needs to be recovered across AZ.
                    is_cross_az_restore = True
                    LOG.info('this is cross different az restore or remote restore.')
            else:
                if image is None:
                    msg = self._convert_error(ErrorCode.IMAGE_NON_EXISTENT,
                                              "get restore image failed, the image does not exist.")
                    LOG.error(msg)
                    raise exception.InvalidBackup(msg)

            if is_cross_az_restore and ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD:
                # remote replication will copy HI and AC to dest region, but BP will be ebackup path
                if metadata.get("HI", 0) and "AC" not in metadata.keys():
                    self._get_public_cloud_storage_info(backup, True, metadata.get("BP"))
                else:
                    self._get_public_cloud_storage_info(backup, True)

            display_name = backup['display_name']
            flag = self._check_length(display_name, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ("display name length does not comply, error length:%d" % len(display_name))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            cur_snapshot_url = volume_file['volume_url']
            if cur_snapshot_url is None:
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, "current snapshot is None")
                LOG.error(error_msg)
                raise Exception(error_msg)

            lastest_snapshot_url = volume_file.get('lastest_snapshot_url', "")
            if lastest_snapshot_url is None:
                lastest_snapshot_url = ""

            LOG.info(str(source_volume_id) + ':Begin hwsebackup restore.')
            vol_offset_size = -1
            source_vol = call_db_fun(self.db.volume_get, self.context, volume_id)
            if str(storage_type) in (cst.STORAGE_TYPE_VRM_FUSION_STORAGE,
                                     cst.STORAGE_TYPE_VRM_FUSION_RDM,
                                     cst.STORAGE_TYPE_KVM_FUSION_STORAGE):
                vol_offset_size_str = self.__get_volume_offset_size_from_provider_location(source_vol)
                if vol_offset_size_str not in (None, ''):
                    vol_offset_size = int(vol_offset_size_str)
            if source_vol.get('vendor_volume_driver_metadata', ''):
                metadata_temp = source_vol.get('vendor_volume_driver_metadata')
                meta_dict = {item['key']: item['value'] for item in metadata_temp}
                if meta_dict.get('xen_clock_offset'):
                    vol_offset_size = int(meta_dict.get('xen_clock_offset'))
                    LOG.info('get migrate offset %s' % meta_dict.get('xen_clock_offset'))

            RequestID = "%d" % hash("%s" % uuid.uuid1())
            if RequestID.startswith('-'):
                RequestID = RequestID[1:]
            LOG.info("RequestID : %s" % RequestID)
            if metadata.get("C-Region"):
                restore_url = self.ebackup_restore_remote_uri
                if metadata.get('BT') in (FILESYS_CIFS, FILESYS_NFS):
                    spath = metadata.get('NPATH')
                else:
                    spath = metadata.get('BP')
                restore_body = {
                    "REQUEST_ID": RequestID,
                    "STORAGE_TYPE": storage_type,
                    "IMAGE_ID": image_id,
                    "SOURCEREGIONID": metadata.get('SrcRegionID'),
                    "VOLUME_NAME": '',
                    "VOLUME_ID": volume_id,
                    "VOLUME_SIZE": volume_size,
                    "VOLUME_URL": cur_snapshot_url,
                    "VOLUME_OFFSET_SIZE": vol_offset_size,
                    "DATA_ENC_INFO": {
                        "CMKID": metadata.get('CMKID', ''),
                        "VK": metadata.get('VK', ''),
                        "DEK": metadata.get('DEK', '')
                    },
                    "ORIGINDATAINFO": {
                        "DATALAYOUT": metadata.get('ODL'),
                        "PROTOCOL": int(metadata.get('PR', 0)),
                        "STORAGE_TYPE": int(metadata.get('BT')),
                        "STORAGE_UNIT_PATH": spath,
                        "IMAGE_PATH": metadata.get('SP'),
                        "AK": metadata.get('AK'),
                        "SK": metadata.get('SK'),
                        "REGIONIP": metadata.get('EBKIP'),
                        "REGIONPORT": int(metadata.get('EBKPORT')),
                        "IAMUSERNAME": metadata.get('IAMUser'),
                        "IAMPASSWORD": Utils.decrypt_password(metadata.get('IAMPassword')),
                        "TENANTID": metadata.get('SrcTenantID'),
                        "DOMAINID": metadata.get('DomainID'),
                        "QOS": metadata.get('BandWidth'),
                        "ACCLERATEOPTION": 1 if metadata.get("NeedAccelerate") else 0,
                    }
                }
            elif is_cross_az_restore:
                restore_url = self.ebackup_restore_direct_uri
                restore_body = {
                    "STORAGE_TYPE": storage_type,
                    "STORAGE_UNIT_TYPE": metadata.get('BT'),
                    "STORAGE_UNIT_PATH": metadata.get('BP'),
                    "IMAGE_PATH": metadata.get('SP'),
                    "BACKUP_DATA_LAYOUT": metadata.get('DL'),
                    "IMAGE_ID": image_id,
                    "RESTORE_VOLUME_ID": volume_id,
                    "BACKUP_VOLUME_SIZE": volume_size,
                    "BACKEDUP_VOLUME_ID": source_volume_id,
                    "LATEST_SNAPSHOT_URL": lastest_snapshot_url,
                    "VOLUME_URL": cur_snapshot_url,
                    "REQUEST_ID": RequestID,
                    "REGION": str(region_name),
                    "VOLUME_OFFSET_SIZE": vol_offset_size,
                    "EBACKUP_GUID": metadata.get('e_u', '')
                }
                restore_body = dict(restore_body, **self.evs_body)
            else:
                restore_url = self.ebackup_restore_disk_uri
                restore_body = {
                    "BACKEDUP_MACHINE_ID": source_volume_id,
                    "BACKEDUP_SNAP_VERSION": str(image_id),
                    "VOLUME_ID": volume_id,
                    "VOLUME_SIZE": volume_size,
                    "VOLUME_URL": cur_snapshot_url,
                    "LATEST_SNAPSHOT_URL": lastest_snapshot_url,
                    "TARGET_LOCATION": {
                        "DiskDestination": [
                            {
                                "8": "",
                                "9": "",
                                "12": ""
                            }
                        ]
                    },
                    "RESTORE_MODE": cst.RESTORE_MODE_OLD,
                    "RESTORE_TYPE": cst.RESTORE_TYPE_VOLUME,
                    "POWERON_AFTER_RESTORE": False,
                    "CHANNEL_TYPE": cst.CHANNEL_TYPE,
                    "STORAGE_TYPE": storage_type,
                    "REQUEST_ID": RequestID,
                    "REGION": str(region_name),
                    "VOLUME_OFFSET_SIZE": vol_offset_size
                }
                restore_body = dict(restore_body, **self.evs_body)
            if vol_offset_size == -1:
                restore_body.pop("VOLUME_OFFSET_SIZE")

            restore_body_info = copy.deepcopy(restore_body)
            restore_body_info['DATA_ENC_INFO']['VK'] = "******"
            restore_body_info['DATA_ENC_INFO']['DEK'] = "******"
            if metadata.get("IT") == cst.IMPORT_TYPE_REMOTE_RESTORE:
                restore_body_info['ORIGINDATAINFO']['IAMPASSWORD'] = "******"
                restore_body_info['ORIGINDATAINFO']['SK'] = "******"
            LOG.info('restore body is:' + str(restore_body_info))
            # write requestID to db
            request_id_s = "R_%s" % RequestID
            self._change_value_by_keys(metadata, ebk_T_I=str(request_id_s))
            self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
            # whether backuping by target volume
            request_id = self._check_backuping_by_restore(restore_url, restore_body)
            task_type_restore = cst.TASKTYPE_VOL_RESTORE
            if metadata.get("C-Region"):
                task_type_restore = cst.TASKTYPE_VOL_REMOTE_RESTORE
            task_id = self._get_task_id(task_type_restore, request_id)

            self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
            self._change_value_by_keys(metadata, AT=0.0)
            self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

            while True:
                time.sleep(cst.TIME_SLEEP_30)
                task_infos = self.get_task_info(task_id)
                if not isinstance(task_infos, dict):
                    LOG.error("task : %s failure,failed to get task info" % task_id)
                    break

                task_status, task_progress, task_details = self._get_task_progress_and_details(task_infos,
                                                                                               task_type_restore)
                LOG.info('restore task progress is:%s' % (task_progress + '%'))
                if task_status == cst.TASK_STATUS_FAILED:
                    msg = ("eBackup Restore VM disk job failed, job id is:" + str(task_id))
                    fail_reason_str = self._get_task_detail(task_details, msg)
                    LOG.error("Restore job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                    raise exception.InvalidBackup(fail_reason_str)

                # update task progress.
                if task_progress == '0':
                    task_progress = '1'
                self._change_value_by_keys(metadata, progress=str(task_progress))
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

                ret_status, task_speed = self._get_task_details_speed(task_infos)
                if not ret_status:
                    msg = "SPEED in restore is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    self.error_info.set_error_info(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)

                if -1 != task_speed:
                    self._change_value_by_keys(metadata, AT=task_speed)
                    self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

                if task_status == cst.TASK_STATUS_COMPLETED:
                    self._change_value_by_keys(metadata, progress='0', ebk_T_I='')
                    self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata),
                                        fail_reason='')
                    break
            LOG.info('%s:End hwsebackup restore.' % source_volume_id)

        except Exception as msg:
            LOG.error(msg)
            LOG.error('Origin stack is:%s' % traceback.format_exc())
            if metadata:
                self._change_value_by_keys(metadata, progress='0')
                self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata),
                                    fail_reason=str(msg))
            else:
                self._update_backup(backup_id, fail_reason=str(msg))
            raise exception.InvalidBackup(str(msg))

    def _remove_keys(self, service_metadata, *args):
        """ Remove the args from service metadata.
        """
        for i in args:
            if i in service_metadata:
                service_metadata.pop(i)

    def _change_value_by_keys(self, service_metadata, **kwargs):
        """ Modify the service_metadata by args, if args dont exist, append it.
        """
        for i in kwargs:
            service_metadata[i] = kwargs[i]

    def _update_backup(self, backup_id, **kwargs):
        """ Update the param by backup id.
        """
        call_db_fun(self.db.backup_update, self.context,
                    backup_id,
                    kwargs)

    @HandleErrorDesc(ErrorCode.BD_DELETE_FAILED)
    def post_delete(self, backup, last_status='pre_delete'):
        try:
            self.error_info.error_code = ErrorCode.BD_DELETE_FAILED
            self.__task_type = 'delete'
            self.__backup = backup
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()

            self.update_om_hostid_for_backup(backup, "")
            self.init_evs_body(backup, 'delete')
            backup_meta = Utils.transfer_json_to_dict(backup.get('service_metadata'))
            if (self.get_import_type(
                    backup) in (cst.IMPORT_TYPE_DIFFERENT_AZ_RESTORE, cst.IMPORT_TYPE_REMOTE_RESTORE)) or self.is_remote_replication_error_backup(backup):
                LOG.info("backup[%s] with import type[%s] delete", backup['id'], backup_meta.get('IT'))
                return
            tenant_id = backup['project_id']
            self.__availability_zone = backup['availability_zone']
            if not self.__availability_zone:
                LOG.info('Not Found availability_zone,image not clear')
                return
            flag = self._check_length(tenant_id, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ('tenant id length does not comply, error length:%d' % len(tenant_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            LOG.info('Begin hwsebackup delete, tenant id:%s' % tenant_id)
            volume_id = self._get_cascaded_volume_id(backup)

            if volume_id == cst.VOLUME_INDEX:
                LOG.info('The volume is index.')
                return
            flag = self._check_length(volume_id, cst.LENGTH_LIMIT_36)
            if flag:
                msg = ('volume id length does not comply, error length:%d' % len(volume_id))
                error_msg = convert_error(ErrorCode.BD_VOLUME_ID_INVALID, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            LOG.info('Begin hwsebackup delete, source volume id:%s' % volume_id)

            backup_id = backup['id']
            if len(backup_id) != cst.LENGTH_LIMIT_36:
                msg = ('backup id length does not comply, error length:%d' % len(backup_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            LOG.info('Begin hwsebackup delete, backup id:%s' % backup_id)

            backup_in_database = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_service_metadata = backup_in_database.get('service_metadata')
        except Exception as msg:
            LOG.error(msg)
            LOG.error('Origin stack is:%s' % traceback.format_exc())
            raise exception.InvalidBackup(msg)

        if json_service_metadata in (None, ''):
            LOG.info("service_metadata is null, backup is error backup, so no need to login eBackup.")
            return

        metadata = json.loads(json_service_metadata)
        self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
        LOG.info("physical network type: %s", self.__physical_network_type)
        useless_key = ('VMID', 'DT', 'AT', 'CT')
        self._remove_keys(metadata, *useless_key)
        image_id = metadata.get('backupurl')
        LOG.info('Begin hwsebackup delete, image id:%s' % image_id)

        osimg = metadata.get('osimg')
        if '1' == osimg or 1 == osimg:
            LOG.info("This backup is using as an os image, can not be deleted.")
            raise Exception(cst.BACKUP_OS_IMAGE)

        self._change_value_by_keys(metadata, progress='0')
        self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

        self._do_delete(backup, last_status)
        LOG.info(volume_id + ':End hwsebackup post delete.')

    def delete_backup(self, backup):
        self.delete(backup)

    @profiler.trace("ebackup driver:invoke to delete backup Image")
    def delete(self, backup):
        """Delete the given backup from eBackup server.
        :param backup: backup information for volume
        :raises InvalidBackup
        """
        try:
            self.__task_type = 'delete'
            self.__backup = backup
            LOG.info("begin delete backup[%s]" % backup['id'])
            self.post_delete(backup, 'available')
            if self.get_import_type(backup) not in (
                    cst.IMPORT_TYPE_NATIVE_TO_CLOUD, cst.IMPORT_TYPE_DIFFERENT_AZ_RESTORE, cst.IMPORT_TYPE_REMOTE_RESTORE) \
                    and not self.is_remote_replication_error_backup(backup):
                try:
                    latest_backup = self._update_backup_total_size(backup)
                    if not latest_backup:
                        latest_backup = dict()
                except Exception:
                    latest_backup = dict()
                    LOG.info("Not update chainsize!")
                cashier = ExpensesEvent(self.db, 'delete')
                cashier.send_expenses_event(self.context, backup, latest_backup, successd=True)
            LOG.info(backup['id'] + ':End hwsebackup delete.')
        except Exception as msg:
            LOG.error(msg)
            latest_backup = self._get_latest_backup_id(backup)
            if not latest_backup:
                latest_backup = dict()
            cashier = ExpensesEvent(self.db, 'delete')
            cashier.send_expenses_event(self.context, backup, latest_backup, successd=False)
            raise exception.InvalidBackup(msg)

    def get_property(self, backup):
        ''' Get backup property'''
        try:
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()
            self.__task_type = 'delete'
            self.__backup = backup
            LOG.info('the backup id is: %s' % backup['id'])
            self.__availability_zone = backup['availability_zone']
            tenant_id = backup['project_id']
            flag = self._check_length(tenant_id, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ('tenant id length does not comply, error length:%d' % len(tenant_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            json_service_metadata = backup.get('service_metadata')
            backupCapacity = {}
            if json_service_metadata in (None, ''):
                LOG.warn("json_service_metadata is empty, treat it as not support async directly")
                backupCapacity['is_support_async_delete'] = False
                return backupCapacity

            # go to execute delete() backup while it is a import backup, it won't be deleted in the delete()
            if self.get_import_type(backup) == cst.IMPORT_TYPE_DIFFERENT_AZ_RESTORE:
                LOG.info("into across az restore delete record from database!")
                backupCapacity['is_support_async_delete'] = False
                return backupCapacity
            elif self.get_import_type(backup) == cst.IMPORT_TYPE_NATIVE_TO_CLOUD:
                LOG.info("into native to cloud property!")
                backupCapacity['is_support_async_delete'] = False
                return backupCapacity

            metadata = json.loads(json_service_metadata)
            self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
            image_id = metadata.get('backupurl')

            if image_id in (None, ''):
                backupCapacity['is_support_async_delete'] = False
                return backupCapacity

            LOG.info('the backup image id is:%s' % image_id)
            image = self._get_image_by_id(image_id, sync=True)

            if image is None:
                backupCapacity['is_support_async_delete'] = False
                return backupCapacity

            snap_version = image.get('VERSION')
            if snap_version not in (None, ''):
                backupCapacity['version'] = snap_version
            else:
                LOG.info('not found backup version in response body.')
            snap_isasync = image.get('ISASYNC', '')
            if snap_isasync == '1':
                backupCapacity['is_support_async_delete'] = True
            else:
                backupCapacity['is_support_async_delete'] = False
            LOG.info('get property %s', backupCapacity)
            return backupCapacity
        except Exception as msg:
            LOG.error("get backup version failed, reason:%s" % msg)
            raise exception.InvalidBackup(msg)

    def pre_delete(self, backup):
        cashier = ExpensesEvent(self.db, 'delete')
        try:
            self.__task_type = 'delete'
            self.__backup = backup
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()
            self.update_om_hostid_for_backup(backup, "")
            if cst.PROVIDER_METADATA in backup.keys():
                self.init_evs_body(backup, 'delete')
            LOG.info('the backup id is: %s' % backup['id'])
            if hasattr(self.context, 'read_deleted'):
                LOG.info("read deleted backup: %s", self.context.read_deleted)
            self.__availability_zone = backup['availability_zone']
            tenant_id = backup['project_id']
            flag = self._check_length(tenant_id, cst.LENGTH_LIMIT_255)
            if flag:
                msg = ('tenant id length does not comply, error length:%d' % len(tenant_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            json_service_metadata = backup.get('service_metadata')
            if json_service_metadata in (None, ''):
                LOG.warn("json_service_metadata is empty, pre delete success")
                return {}

            # do not calc size while deleting backup from cross az restore
            if self.get_import_type(backup) in (cst.IMPORT_TYPE_DIFFERENT_AZ_RESTORE, cst.IMPORT_TYPE_NATIVE_TO_CLOUD) \
                    or self.is_remote_replication_error_backup(backup):
                LOG.info("into across az restore or remote copy delete record or cloud native delete from database!")
                return {}

            metadata = json.loads(json_service_metadata)
            self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
            image_id = metadata.get('backupurl')
            if image_id in (None, ''):
                LOG.warn("backupurl doesn't exist")
                return {}

            LOG.info('the backup image id is:%s' % image_id)
            RequestID = "%d" % hash("%s" % uuid.uuid1())
            if RequestID.startswith('-'):
                RequestID = RequestID[1:]
            LOG.info("RequestID : %s" % RequestID)
            pre_delete_body = {'PREDELETE': True}
            delete_result = self._delete_image_object(self.ebackup_image_uri, image_id, RequestID, pre_delete_body, retry=False, sync=True)
            error_number = self._get_error(delete_result)
            if error_number == ErrorCode.IMAGE_NON_EXISTENT:
                LOG.info('The backup(ID:%s, image ID:%s) was deleted by eBackup.', backup['id'], image_id)
            elif error_number == ErrorCode.IMAGE_DELETING:
                LOG.info('The backup(ID:%s, image ID:%s) was deleting by eBackup.', backup['id'], image_id)
            elif error_number != 0:
                error_msg = convert_error(error_number, 'delete backup image fail')
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)

            need_update_chainsize = False
            backupCapacity = {}
            image = self._get_image_by_id(image_id)
            if image is None:
                need_update_chainsize = True
            else:
                snap_version = image.get('VERSION')
                if snap_version not in (None, ''):
                    backupCapacity['version'] = snap_version
                else:
                    LOG.info('not found backup version in response body.')
                snap_isasync = image.get('ISASYNC', '')
                if snap_isasync == '1':
                    backupCapacity['is_support_async_delete'] = True
                else:
                    backupCapacity['is_support_async_delete'] = False
                if snap_isasync not in (None, '') and snap_isasync == '1':
                    snap_domain_id = image.get('ASSOCIATEBACKUPDOMAINNAME')
                    if snap_domain_id in (None, ''):
                        msg = self._convert_error(ErrorCode.IMAGE_NON_EXISTENT,
                                                  "get BACKUPDOMAINNAME failed when pre delete backup")
                        LOG.error(msg)
                        raise exception.InvalidBackup(msg)
                    need_update_chainsize = True
            if need_update_chainsize:
                domain_type = cst.DOMAIN_TYPE_BACKUP
                if 'IT' in json_service_metadata:
                    domain_type = metadata['IT']
                volume_backup_size, update, ad_flag = self._get_total_chain_size(backup.get('volume_id'),
                                                                                 domain_type,
                                                                                 metadata.get("ST"),
                                                                                 backup)
                latest_backup = self._get_latest_backup_id(backup)
                try:
                    if latest_backup is not None:
                        latest_backup['object_count'] = volume_backup_size
                        self._update_backup(latest_backup.get('id'), object_count=volume_backup_size)
                        ExpensesEvent.sync_backup_to_cascading(latest_backup.get('display_name', ''), 'object_count',
                                                               volume_backup_size)
                        if ad_flag:
                            latest_backup_meta = Utils.transfer_json_to_dict(latest_backup.get('service_metadata'))
                            self._change_value_by_keys(latest_backup_meta, AD=1)
                            latest_backup['service_metadata'] = Utils.transfer_to_json_str(latest_backup_meta)
                            self._update_backup(latest_backup.get('id'),
                                                service_metadata=Utils.transfer_to_json_str(latest_backup_meta))
                            ExpensesEvent.sync_backup_to_cascading(latest_backup.get('display_name', ''),
                                                                   'service_metadata',
                                                                   Utils.transfer_to_json_str(latest_backup_meta))
                        LOG.info('change [%s] chainsize to %s', latest_backup.get('id'), volume_backup_size)
                    else:
                        LOG.info('latest backup id is None, it will not update backup size.')
                except Exception as umsg:
                    LOG.info('update last backup failed. maybe it is not exist now %s' % umsg)
                cashier.send_expenses_event(self.context, backup, latest_backup, successd=True)
            LOG.info('get property %s', backupCapacity)
            return backupCapacity
        except Exception as msg:
            latest_backup = self._get_latest_backup_id(backup)
            if not latest_backup:
                latest_backup = dict()
            cashier.send_expenses_event(self.context, backup, latest_backup, successd=False)
            LOG.error("get backup version failed, reason:%s" % msg)
            raise exception.InvalidBackup(msg)

    def _update_copy_volume_id(self, backup, backup_id):
        if cst.TASKTYPE_COPY != self.__driver_task_type and cst.TASKTYPE_REMOTE_REPLICATION != self.__driver_task_type:
            return
        if self.__copy_image_volume_id in (None, ''):
            return
        backup['volume_id'] = self.__copy_image_volume_id
        self._update_backup(backup_id, volume_id=self.__copy_image_volume_id, status=cst.BACKUP_STATUS_CREATING)
        LOG.info("update copy volume id(%s) by backup id(%s)." % (self.__copy_image_volume_id, backup_id))
        self.__copy_image_volume_id = ''

    def _update_copied_volume_id_and_progress(self, backup_id):
        try:
            copy_image_volume_id = cst.VOLUME_INDEX
            if self.__copy_image_volume_id not in (None, ''):
                copy_image_volume_id = self.__copy_image_volume_id

            backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            metadata = json.loads(backup.get('service_metadata'))
            self._change_value_by_keys(metadata, progress='0')
            self._update_backup(backup_id, service_metadata=json.dumps(metadata), volume_id=copy_image_volume_id)
        except Exception as msg:
            self._update_backup(backup_id, service_metadata='None')
            LOG.error("update copy task fail, reason:%s" % msg)
            raise exception.InvalidBackup(str(msg))

    def native_copy(self, backup, backup_id):
        try:
            if self.__backup_info:
                LOG.info("this is an continuous copy task")
                backup_from_import = self.__backup_info
                backup_from_import['id'] = backup_id
            else:
                LOG.info("this is an copy task after interrupt")
                backup_from_import = backup
            self._do_import_record(backup_from_import, backup_id)
            # import were update backup,so we need reload it
            backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            backup_status = backup.get('status')
            backup_fail_reason = backup.get('fail_reason')
            if backup_status == cst.BACKUP_STATUS_ERROR:
                if backup_fail_reason == cst.EBACKUP_DRIVER_RESTART:
                    self._update_backup(backup_id, status=cst.BACKUP_STATUS_CREATING,
                                        fail_reason='')
            elif backup_status == cst.BACKUP_STATUS_AVALIBLE:
                LOG.info('the backup status is avaliable.')
                return
            elif backup_status != cst.BACKUP_STATUS_CREATING:
                error_msg = convert_error(ErrorCode.BD_BACKUP_STATE_NOT_RIGHT, 'backup status is not correct')
                LOG.error('backup status(%s) is not correct' % backup_status)
                raise Exception(error_msg)
            LOG.info('Begin hwsebackup verify, backup_status :%s' % backup_status)

            json_service_metadata = backup.get('service_metadata')
            if json_service_metadata in (None, ''):
                msg = "service_metadata is null, verify failed."
                LOG.error(msg)
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)

            metadata = json.loads(json_service_metadata)
            task_id_metadata = metadata.get('ebk_T_I')
            if task_id_metadata in (None, ''):
                msg = "ebk_T_I in service_metadata is null, verify failed."
                LOG.error(msg)
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)

            LOG.info('Begin hwsebackup verify, ebk_T_I :%s' % task_id_metadata)
            task_id = ''
            request_id = ''
            task_type = cst.TASKTYPE_COPY

            if task_id_metadata.startswith('R_'):
                request_id = task_id_metadata[2:]
            else:
                task_id = task_id_metadata

            if task_id in (None, ''):
                task_id = self._get_task_id(task_type, request_id)
                self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
                self._update_backup(backup_id, service_metadata=json.dumps(metadata), created_at=timeutils.utcnow())

            while True:
                time.sleep(cst.TIME_SLEEP_30)
                task_infos = self.get_task_info(task_id)
                if not isinstance(task_infos, dict):
                    LOG.error("task : %s failure,failed to get task info" % task_id)
                    raise Exception("failed to get task {task_id} info".format(
                        task_id=task_id))

                task_status, task_progress, task_details = self._get_task_progress_and_details(task_infos, task_type)
                task_speed_kb = task_infos.get("SPEED")
                if task_speed_kb in (None, ''):
                    msg = "SPEED in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)
                task_space_saving_rate = task_infos.get("SPACESAVINGRATE")
                if task_space_saving_rate in (None, ''):
                    msg = "SPACESAVINGRATE in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)
                task_snap_type = task_infos.get("SNAPTYPE")
                if task_snap_type in (None, ''):
                    msg = "SNAPTYPE in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)

                MbToKb = cst.Mi / cst.Ki
                task_speed_mb = float(task_speed_kb) / MbToKb

                dict_container_info = {}
                json_container = backup.get('container')
                if json_container in (None, ''):
                    dict_container_info["AT"] = round(task_speed_mb, 1)
                    dict_container_info["SS"] = int(task_space_saving_rate)
                else:
                    if -1 == json_container.find(":"):  # J old data
                        dict_container_info["AT"] = round(task_speed_mb, 1)
                        dict_container_info["SS"] = int(task_space_saving_rate)
                        dict_container_info["SNAPID"] = json_container
                    else:
                        dict_container_info = eval(json_container)
                        at_metadata = dict_container_info.get("AT")
                        if at_metadata in (None, ''):
                            msg = "AT in container is null, verify failed."
                            LOG.error(msg)
                            error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                            raise exception.InvalidBackup(error_msg)
                        else:
                            dict_container_info["AT"] = round(task_speed_mb, 1)

                        ss_metadata = dict_container_info.get("SS")
                        if ss_metadata in (None, ''):
                            msg = "SS in container is null, verify failed."
                            LOG.error(msg)
                            error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                            raise exception.InvalidBackup(error_msg)
                        else:
                            dict_container_info["SS"] = int(task_space_saving_rate)

                self._update_backup(backup_id, container=json.dumps(dict_container_info))
                self._change_value_by_keys(metadata, Type=int(task_snap_type))
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))

                LOG.info('verify a task progress is:%s' % task_progress)
                # backup failed
                if task_status == cst.TASK_STATUS_FAILED:
                    msg = "task failed, eBackup task id is:%s" % task_id
                    fail_reason_str = self._get_task_detail(task_details, msg)
                    LOG.error("Job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                    raise exception.InvalidBackup(fail_reason_str)

                self._change_value_by_keys(metadata, progress=str(task_progress))
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))

                # backup completed
                if task_status == cst.TASK_STATUS_COMPLETED:
                    if self.__copy_image_source_id in (None, ''):
                        error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, "the source image id is None")
                        LOG.error(error_msg)
                        raise exception.InvalidBackup(error_msg)
                    LOG.info("get copy image id by backup image id:%s" % str(self.__copy_image_source_id))
                    copy_snap_guid_list, _N_ = self._get_copy_snap_guids_by_snap_id(self.__copy_image_source_id,
                                                                                    str(backup["volume_id"]))
                    if len(copy_snap_guid_list) <= 0:
                        # if image id is null, it means copy failed.
                        error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, "get new copy image id is None")
                        LOG.error(error_msg)
                        raise exception.InvalidBackup(error_msg)
                    image_id = copy_snap_guid_list[0]

                    if image_id is None:
                        # if image id is null, it means backup failed.
                        error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, 'get new image id is None')
                        LOG.error(error_msg)
                        raise exception.InvalidBackup(error_msg)
                    snap_path = task_infos.get('IMAGEPATH', "")
                    if not snap_path:
                        msg = "snap path is empty, task failed"
                        LOG.error(msg)
                        error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                        raise exception.InvalidBackup(error_msg)
                    # get volume backup size (totle size)
                    domain_type = cst.DOMAIN_TYPE_BACKUP
                    if 'IT' in metadata:
                        domain_type = metadata['IT']
                    volume_backup_size, update, ad_flag = self._get_total_chain_size(self._get_cascaded_volume_id(backup),
                                                                                     domain_type,
                                                                                     metadata.get('ST'),
                                                                                     backup)
                    self._remove_keys(metadata, "backupurl_tmp")
                    self._change_value_by_keys(metadata, backupurl=str(image_id), progress='0',
                                               CS=int(volume_backup_size), SP=snap_path, ebk_T_I="")
                    self._update_backup(backup_id, object_count=volume_backup_size,
                                        service_metadata=json.dumps(metadata),
                                        availability_zone=backup["availability_zone"],
                                        status=cst.BACKUP_STATUS_AVALIBLE)
                    break
        except Exception as msg:
            if backup is None:
                backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_service_metadata = backup.get('service_metadata')
            metadata = Utils.transfer_json_to_dict(json_service_metadata)
            self._change_value_by_keys(metadata, progress='0')
            self._update_backup(backup_id,
                                status=cst.BACKUP_STATUS_ERROR,
                                service_metadata=Utils.transfer_to_json_str(metadata),
                                fail_reason=str(msg))
            LOG.info("copy task failed: %s,update copy task status to error", msg)
            raise exception.InvalidBackup(msg)

    def setup_remote_replication_backup_info(self, backup, import_backup, copy_context, e_uuid):
        tenant_id = import_backup['project_id']
        flag = self._check_length(tenant_id, cst.LENGTH_LIMIT_255)
        if flag:
            msg = ('tenant id length does not comply, error length:%d' % len(tenant_id))
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
            LOG.error(error_msg)
            raise Exception(error_msg)
        LOG.info('Begin copy, copy id:%s' % import_backup['id'])
        metadata = Utils.transfer_json_to_dict(import_backup['service_metadata'])
        # store some basice info to backup for redo remote copy
        temp_backup = {}
        temp_metadata = {"ST": metadata.get('ST'), "backupurl_tmp": metadata.get('backupurl_tmp'), "e_u": e_uuid,
                         "OMID": Utils.get_hostname()}
        self._change_value_by_keys(metadata, OMID=Utils.get_hostname())
        self.update_copy_flag(temp_backup, temp_metadata, cst.DOMAIN_TYPE_REMOTE_REPLICATION)
        self._update_backup(backup['id'], service_metadata=Utils.transfer_to_json_str(temp_metadata),
                            created_at=timeutils.utcnow())
        metadata.update(temp_metadata)
        # fetch pnt value for get_connection
        self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
        LOG.info("physical network type: %s", self.__physical_network_type)
        backup_image_id = metadata['backupurl_tmp']
        copy_context['backup_image_id'] = backup_image_id
        self.__copy_image_source_id = backup_image_id
        if metadata.get('CMKID') and metadata.get('VK') and metadata.get("DEK"):
            evs = EVS()
            copy_context['CMKID'], copy_context['VK'], copy_context['DEK'] = \
                evs.get_replication_body_if_backup_is_encrypt(self.db,
                                                              self.context,
                                                              backup.get('project_id'),
                                                              backup.get('volume_id'))
            if not copy_context['VK'] and not copy_context['DEK']:
                copy_context['CMKID'] = EVS.get_replication_cmk_id(self.context, "vbs/default",
                                                                   backup.get('project_id'))
                copy_context['VK'], copy_context['DEK'] = evs.get_remote_replication_body(self.db, self.context,
                                                                                          copy_context['CMKID'],
                                                                                          backup.get('project_id'),
                                                                                          backup.get('volume_id'))
            if not copy_context['VK'] or not copy_context['DEK']:
                LOG.error("vk and dek is invalid:%s, %s", copy_context['VK'], copy_context['DEK'])
                raise exception.InvalidBackup("vk and dek is invalid")
            copy_context['OCMKID'] = metadata['CMKID']
            copy_context['OVK'] = metadata['VK']
            self._change_value_by_keys(metadata,
                                       CMKID=copy_context['CMKID'],
                                       VK=copy_context['VK'],
                                       OCMKID=copy_context['OCMKID'],
                                       OVK=copy_context['OVK'],
                                       DEK=metadata.get("DEK"))
        else:
            evs = EVS()
            copy_context['CMKID'] = "vbs/default"
            copy_context['CMKID'], last_backup_vk, last_backup_dek = \
                evs.get_replication_body_if_backup_is_encrypt(self.db,
                                                              self.context,
                                                              backup.get('project_id'),
                                                              backup.get('volume_id'))
            if last_backup_vk and last_backup_dek:
                copy_context['VK'], copy_context['DEK'] = last_backup_vk, last_backup_dek
                self._change_value_by_keys(metadata,
                                           CMKID=copy_context['CMKID'],
                                           VK=copy_context['VK'])

        if copy_context.get('CMKID') and copy_context.get('VK') and \
                copy_context.get("DEK"):
            pro_meta = dict()
            pro_meta[cst.CMKID_KEY] = copy_context['CMKID']
            pro_meta[cst.VK_KEY] = copy_context['VK']
            pro_meta[cst.DEK_KEY] = Utils.encrypt_password(
                copy_context.get("DEK"))

            self._update_backup(
                backup['id'],
                provider_metadata=Utils.transfer_to_json_str(pro_meta))
            LOG.info("Update vk and dek to provider_metadata for backup:"
                     "{id}".format(id=backup['id']))

        backup['service_metadata'] = Utils.transfer_to_json_str(metadata)

    def get_ebk_task_id(self, metadata, task_type):
        if not metadata.get("ebk_T_I"):
            return False, ""
        previous_task_id = metadata.get("ebk_T_I")
        task_id = ''
        request_id = ''
        if previous_task_id.startswith('R_'):
            request_id = previous_task_id[2:]
        else:
            task_id = previous_task_id
        try:
            if task_id in (None, ''):
                task_id = self._get_task_id(task_type, request_id)
            return True, task_id
        except Exception as msg:
            LOG.error("previous task query failed with %s" % msg)
            return False, request_id

    def create_remote_replication_resource(self, backup, copy_context):
        self.create_or_update_copy_policy()
        self._create_copy_related_resource(backup, "")
        # get copy plan ID
        rep_plan_name = self.get_remote_replication_plan_name(backup)
        ebk_client = EbackupConnectionClient(self.__get_connection())
        replication_plan_list, error_code, error_msg = ebk_client.get_copy_plan_by_name(rep_plan_name)
        if error_code != 0 or not replication_plan_list or 'ID' not in replication_plan_list[0].keys():
            msg = self._convert_error(ErrorCode.COPY_PLAN_NON_EXISTENT, "this copy plan does not exist.")
            LOG.error(msg)
            raise exception.InvalidBackup(msg)

        # get copy plan id from backup server
        replication_plan = replication_plan_list[0]
        copy_plan_id = replication_plan.get('ID')
        copy_context['copy_plan_id'] = copy_plan_id

        self.__backup_orig_id = backup['id']
        metadata = Utils.transfer_json_to_dict(backup['service_metadata'])
        storage_path = metadata.get('BP') + '/' + metadata.get('SP')
        copy_plan_body = {}

        if ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD:
            copy_plan_body = {
                "ID": copy_plan_id,
                "TYPE": COPY_PLAN_OBJ_ID,
                "BACKUPIMAGEID": copy_context['backup_image_id'],
                "REQUEST_ID": Utils.gen_request_id(),
                "ORIGINREGIONDATA": {
                    "ORIGINREGIONID": metadata.get('SrcRegionID'),
                    "ORIGINREGIONIP": metadata.get('EBKIP'),
                    "ORIGINREGIONPORT": metadata.get('EBKPORT'),
                    "IAMUSERNAME": metadata.get('IAMUser'),
                    "IAMPASSWORD": metadata.get('IAMPassword'),
                    "ORIGINTENANTID": metadata.get('SrcTenantID'),
                    "ORIGINDOMAINID": metadata.get('DomainID'),
                    "ORIGINSTORAGETYPE": int(metadata.get('BT', FILESYS_S3)),
                    "ORIGINSTORAGEPATH": storage_path,
                    "TARGETVOLUMEGUID": backup['volume_id'],
                    "ORIGINPROTOCOL": int(metadata.get('PR', 0)),
                    "ORIGINBACKUPPLANID": metadata.get('OBPID'),
                    "ORIGINDATALAYOUT": metadata.get('ODL'),
                    "ORIGINAK": metadata.get("AK"),
                    "ORIGINSK": metadata.get("SK"),
                    "ACCLERATEOPTION": 1 if metadata.get("NeedAccelerate") else 0,
                    "QOS": metadata.get("BandWidth")
                },
            }
        elif ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD:
            if metadata.get('BT') == FILESYS_CIFS or metadata.get('BT') == FILESYS_NFS:
                storage_path = metadata.get('NPATH', "") + '/' + metadata.get('SP')
            copy_plan_body = {
                "ID": copy_plan_id,
                "TYPE": COPY_PLAN_OBJ_ID,
                "BACKUPIMAGEID": copy_context['backup_image_id'],
                "REQUEST_ID": Utils.gen_request_id(),
                "ORIGINREGIONDATA": {
                    "ORIGINREGIONID": metadata.get('SrcRegionID'),
                    "ORIGINREGIONIP": metadata.get('EBKIP'),
                    "ORIGINREGIONPORT": metadata.get('EBKPORT'),
                    "IAMUSERNAME": metadata.get('IAMUser'),
                    "IAMPASSWORD": metadata.get('IAMPassword'),
                    "ORIGINTENANTID": metadata.get('SrcTenantID'),
                    "ORIGINDOMAINID": metadata.get('DomainID'),
                    "ORIGINSTORAGETYPE": int(metadata.get('BT', FILESYS_S3)),
                    "ORIGINSTORAGEPATH": storage_path,
                    "TARGETVOLUMEGUID": self._get_cascaded_volume_id(backup),
                    "ORIGINPROTOCOL": int(metadata.get('PR', 0)),
                    "ORIGINBACKUPPLANID": metadata.get('OBPID'),
                    "ORIGINDATALAYOUT": metadata.get('ODL'),
                    "ORIGINAK": metadata.get("AK"),
                    "ORIGINSK": metadata.get("SK"),
                    "ACCLERATEOPTION": 1 if metadata.get("NeedAccelerate") else 0,
                    "QOS": metadata.get("BandWidth"),
                    "TENANTISOLATION": 0
                },
            }

        if metadata.get("DEK"):
            copy_plan_body["ORIGINREGIONDATA"]["ORIGINDATAENCINFO"] = {
                "CMKID": metadata.get("OCMKID", ""),
                "VK": metadata.get("OVK", ""),
                "DEK": metadata.get("DEK", "")
            }
        if copy_context.get("DEK"):
            copy_plan_body["ORIGINREGIONDATA"]["DATAENCINFO"] = {
                "CMKID": metadata.get("CMKID", ""),
                "VK": metadata.get("VK", ""),
                "DEK": copy_context["DEK"]
            }

        copy_context['copy_plan_body'] = copy_plan_body
        debug_copy_body = copy.deepcopy(copy_plan_body)
        for i in ["ORIGINDATAENCINFO", "DATAENCINFO", "IAMPASSWORD", "ORIGINSK"]:
            debug_copy_body["ORIGINREGIONDATA"].pop(i, None)

        copy_snap_guid_list, _N_ = self._get_copy_snap_guids_by_snap_id(metadata['backupurl_tmp'], backup['volume_id'],
                                                                        True)
        if len(copy_snap_guid_list) != 0:
            LOG.info("this backup image has been copied, go to query task info.")
            copy_image_info = self._query_image_detail_by_id(copy_snap_guid_list[0])
            task_id = copy_image_info['JOBID']
            LOG.info('ran task id: %s' % task_id)
        else:
            LOG.info("copy plan body: %s" % debug_copy_body)
            task_id = self.run_copy_plan(copy_plan_body, cst.TASKTYPE_REMOTE_REPLICATION)
        self._change_value_by_keys(metadata, SR=metadata.get('SrcRegionID'))
        self._remove_keys(metadata, *REMOTE_REPLICATION_POP_KEY)
        LOG.info("pop up metadata length:%s, %s", len(metadata), metadata)
        self.update_copy_snap_info(backup, metadata, replication_plan)
        self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
        self._update_backup(backup['id'], service_metadata=Utils.transfer_to_json_str(metadata))
        backup['service_metadata'] = Utils.transfer_to_json_str(metadata)

    def monitor_replication_task(self, backup, copy_context):
        metadata = Utils.transfer_json_to_dict(backup.get('service_metadata'))
        task_id = ''
        for i in range(0, cst.TASK_RETRY_COUNT):
            task_retry_count = i
            result, task_id = self.get_ebk_task_id(metadata, cst.TASKTYPE_REMOTE_REPLICATION)
            if not result and copy_context['continuous_task'] and task_retry_count < cst.TASK_RETRY_COUNT:
                LOG.info("get task failed, try run copy task again")
                task_id = self.run_copy_plan(copy_context['copy_plan_body'], cst.TASKTYPE_REMOTE_REPLICATION)
                continue
            elif not result:
                msg = ("failed to get task info, task id: %s" % task_id)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            else:
                break
        backup_id = backup['id']
        LOG.info('monitor backup %s, ebk_T_I :%s' % (backup_id, task_id))

        while True:
            time.sleep(cst.TIME_SLEEP_30)
            task_infos = self.get_task_info(task_id)
            if not isinstance(task_infos, dict):
                msg = ("task : %s failure,failed to get task info" % task_id)
                LOG.error(msg)
                raise exception.InvalidBackup(msg)
            task_status, task_progress, task_details = \
                self._get_task_progress_and_details(task_infos, cst.TASKTYPE_REMOTE_REPLICATION)
            task_progress = '1' if task_progress == '0' else task_progress
            self._change_value_by_keys(metadata, progress=task_progress)
            LOG.info('progress:%s' % task_progress)
            ret_status, task_speed = self._get_task_details_speed(task_infos)
            if not ret_status:
                msg = "SPEED in backup is null, verify failed."
                LOG.error(msg)
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                self.error_info.set_error_info(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)

            if -1 != task_speed:
                self._change_value_by_keys(metadata, AT=task_speed)

            backup['service_metadata'] = Utils.transfer_to_json_str(metadata)
            self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))

            if task_status == cst.TASK_STATUS_FAILED:
                msg = "task failed, eBackup task id is:%s" % task_id
                fail_reason_str = self._get_task_detail(task_details, msg)
                LOG.error("Job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                raise exception.InvalidBackup(fail_reason_str)

            if task_status == cst.TASK_STATUS_COMPLETED:
                # verify a backup task
                task_snap_type = task_infos.get("SNAPTYPE")
                if task_snap_type in (None, ''):
                    msg = "SNAPTYPE in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)
                self._change_value_by_keys(metadata, Type=int(task_snap_type))
                task_space_saving_rate = self._get_task_details_spacesavingrate(task_infos)
                if task_space_saving_rate in (None, ''):
                    msg = "SPACESAVINGRATE in backup is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)
                self._change_value_by_keys(metadata, SS=int(task_space_saving_rate))
                LOG.info("get copy image id by backup image id:%s" % str(metadata['backupurl_tmp']))
                copy_snap_guid_list = self._get_copy_snaps(metadata['backupurl_tmp'])
                if copy_snap_guid_list is None or len(copy_snap_guid_list) <= 0:
                    # if image id is null, it means copy failed.
                    error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, "get new copy image id is None")
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)
                image_id = copy_snap_guid_list[0]['TARGETSNAPGUID']
                transfer_data = copy_snap_guid_list[0]['TCPTRANSFERDDATA'] + copy_snap_guid_list[0]['VPPTRANSFERDDATA']
                is_accelerate = \
                    int(copy_snap_guid_list[0]['VPPTRANSFERDDATA'] * 100 / transfer_data >= cst.ACCERLERATERATE) \
                        if transfer_data != 0 else 0

                if image_id is None:
                    # if image id is null, it means backup failed.
                    error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, 'get new image id is None')
                    LOG.error(error_msg)
                    raise exception.InvalidBackup(error_msg)

                snap_path = task_infos.get('IMAGEPATH', "")
                LOG.info("remote copy sp:%s", snap_path)
                if not snap_path:
                    msg = "snap path is empty, task failed"
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)
                self._remove_keys(metadata, "backupurl_tmp", "OMID")
                self._change_value_by_keys(metadata, backupurl=str(image_id), progress='0', DT=transfer_data,
                                           SP=snap_path, AC=is_accelerate, CT=int(time.time()))
                self._update_backup(
                    backup_id,
                    service_metadata=Utils.transfer_to_json_str(metadata),
                    availability_zone=backup["availability_zone"])
                backup['service_metadata'] = Utils.transfer_to_json_str(metadata)
                break

    def remote_copy(self, backup, backup_id):
        cashier = ExpensesEvent(self.db, 'copy')
        try:
            LOG.info("remote replication start up, backup id:%s, project_id:%s", backup_id, backup['project_id'])
            copy_context = dict()  # it is used for storing data in task flow
            continuous_task = True
            metadata = Utils.transfer_json_to_dict(backup['service_metadata'])
            if not self.__backup_info:
                LOG.info("this is continuous remote replication")
                if metadata.get("ebk_T_I"):
                    LOG.info("go on monitoring this task: %s" % metadata.get("ebk_T_I"))
                    continuous_task = False
                else:
                    LOG.error("task had not been created in the beginning, remote copy failed")
                    raise exception.InvalidBackup(cst.VERIFY_TASK_NOT_FOUND)
            else:
                backup_from_import = self.__backup_info
                backup_from_import['id'] = backup_id
            copy_context['continuous_task'] = continuous_task
            self.__availability_zone = backup['availability_zone']
            self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
            if continuous_task:
                tmp_backup = call_db_fun(self.db.backup_get, self.context, backup_id)
                tmp_metadata = self.__get_backup_metadata(tmp_backup)
                self._remove_keys(tmp_metadata, "e_u")
                self._update_backup(
                    backup_id,
                    service_metadata=Utils.transfer_to_json_str(tmp_metadata))
                LOG.info("pop e_u for remote-replication: %s" % Utils.transfer_to_json_str(tmp_metadata))
                connTest = self.__get_connection()
                self.setup_remote_replication_backup_info(backup, backup_from_import, copy_context, connTest.ebackup_conf.uuid)
                self.create_remote_replication_resource(backup, copy_context)
            self.monitor_replication_task(backup, copy_context)
            metadata = Utils.transfer_json_to_dict(backup['service_metadata'])
            copy_snap_total_size, update, ad_flag = self._get_total_chain_size(
                self._get_cascaded_volume_id(backup),
                cst.DOMAIN_TYPE_REMOTE_REPLICATION,
                metadata.get('ST'), backup)
            if ad_flag:
                self._change_value_by_keys(metadata, AD=1)
            self._change_value_by_keys(metadata, CS=int(copy_snap_total_size), ebk_T_I="")

            # e_u will be set in __get_connection method
            _backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_service_metadata = _backup.get('service_metadata')
            _metadata = json.loads(json_service_metadata)
            _e_u = _metadata.get('e_u')
            self._change_value_by_keys(metadata, e_u=_e_u)

            self._update_backup(
                backup_id,
                object_count=copy_snap_total_size,
                service_metadata=Utils.transfer_to_json_str(metadata),
                status=cst.BACKUP_STATUS_AVALIBLE)
            backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            cashier.send_expenses_event(self.context, backup, backup, successd=True)
        except Exception as msg:
            with excutils.save_and_reraise_exception():
                json_service_metadata = backup.get('service_metadata')
                metadata = Utils.transfer_json_to_dict(json_service_metadata)
                self._remove_keys(metadata, *REMOTE_REPLICATION_POP_KEY)
                self._change_value_by_keys(metadata, progress='0', IT=cst.IMPORT_TYPE_REMOTE_REPLICATION,
                                           TS=cst.BACKUP_STATUS_ERROR)
                LOG.info("poped metadata length:%s, %s", len(metadata), metadata)
                self._update_backup(backup_id,
                                    status=cst.BACKUP_STATUS_ERROR,
                                    service_metadata=Utils.transfer_to_json_str(metadata),
                                    fail_reason=str(msg))
                backup = call_db_fun(self.db.backup_get, self.context, backup_id)
                cashier.send_expenses_event(self.context, backup, backup, successd=False)

    def is_current_host_backup(self, backup):
        metadata = Utils.transfer_json_to_dict(backup.get('service_metadata'))
        om_host_id = metadata.get("OMID", "")
        node_uuid = Utils.get_hostname()
        if om_host_id == "" or om_host_id == node_uuid:
            LOG.info('start continue backup[%s] on this node[%s][%s]', backup['id'], node_uuid, om_host_id)
            return True
        else:
            LOG.info('the om_host_id is [%s].dont redo on this node[%s]', om_host_id, backup['id'])
            return False

    def remote_restore_verify(self, backup_id):
        LOG.info("update cinder db import type with remote restore.")
        metadata = json.loads(self.__backup_info['service_metadata'])
        self._change_value_by_keys(metadata, IT=cst.IMPORT_TYPE_REMOTE_RESTORE, backupurl=metadata.get("backupurl_tmp"),
                                   IAMPassword=Utils.encrypt_password(metadata.get("IAMPassword")))
        self._remove_keys(metadata, "backupurl_tmp","DL","CMKID","OBPID","VK","AT","SNAPSHOTSIZE","SAZ")
        self._update_backup(backup_id, service_metadata=json.dumps(metadata), object_count=-1)

    @profiler.trace("ebackup driver:invoke to verify backup Image")
    @HandleErrorDesc(ErrorCode.BD_VERIFY_FAILED)
    def verify(self, backup_id):
        self.error_info.error_code = ErrorCode.BD_VERIFY_FAILED
        # restore backup to another availability_zone,restore native backup to cloud
        # we should ignore verify steps
        if self.__driver_task_type in (cst.TASKTYPE_DIRECT_RESTORE, cst.TASKTYPE_NATIVE_TO_CLOUD):
            self.recove_backupurl(backup_id, self.__driver_task_type)
            return

        if self.__driver_task_type is cst.TASKTYPE_VOL_REMOTE_RESTORE:
            self.remote_restore_verify(backup_id)
            return
        # if image has been copied,we should ignore verify steps
        if cst.TASKTYPE_COPY == self.__driver_task_type and self.__is_image_has_copied:
            self._update_copied_volume_id_and_progress(backup_id)
            LOG.info("this backup(id:%s) has been copied, no need to verify." % backup_id)
            return

        # check driver config and income parameter validity
        try:
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()
            if len(backup_id) != cst.LENGTH_LIMIT_36:
                msg = ('backup id length does not comply, error length:%d' % len(backup_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)
            backup_id_msg = ('Begin hwsebackup verify, backup id:%s' % backup_id)
            LOG.info(backup_id_msg)
        except Exception as msg:
            LOG.error(msg)
            raise exception.InvalidBackup(str(msg))

        try:
            backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            backup_status = backup.get('status')
            backup_fail_reason = backup.get('fail_reason')
            backup_metadata = Utils.transfer_json_to_dict(backup.get('service_metadata'))
            if not self.__driver_import_type and not self.arbitration_node(backup_metadata, backup_id):
                return
        except Exception as msg:
            LOG.error("verify failed by %s" % msg)
            return
        self.__task_type = 'copy'
        self.__backup = backup
        # if image was error on osimg delete,we should ignore verify steps
        if backup_status == cst.BACKUP_STATUS_ERROR:
            # clean error msg
            if backup_fail_reason == cst.BACKUP_OS_IMAGE:
                self._update_backup(backup_id, fail_reason='None')
                LOG.info(
                    'backup as an image was changed to error when it would be deleted.clean fail reason on verify and return!')
                return

        if (self.__driver_task_type not in (cst.TASKTYPE_COPY, cst.TASKTYPE_REMOTE_REPLICATION)) and \
                not backup_metadata.get("backupurl_tmp"):  # only processing copy have backupurl_tmp
            # only surport M version now
            LOG.info("verify_for_M begin.")
            self.verify_for_M(backup)
            LOG.info("verify_for_M end.")
            return
        else:
            try:
                self._update_copy_volume_id(backup, backup_id)  # update the volume id of copy image.
                if cst.TASKTYPE_COPY == self.__driver_task_type or backup_metadata.get('IT') == cst.IMPORT_TYPE_COPY:
                    self.update_om_hostid_for_backup("", backup_id, True)
                    self.native_copy(backup, backup_id)
                elif cst.TASKTYPE_REMOTE_REPLICATION == self.__driver_task_type \
                        or backup_metadata.get('IT') == cst.IMPORT_TYPE_REMOTE_REPLICATION:
                    self.remote_copy(backup, backup_id)
                else:
                    error_msg = convert_error(ErrorCode.BD_VERIFY_FAILED, "verify fail while redo copy task")
                    self._update_backup(backup_id,
                                        status=cst.BACKUP_STATUS_ERROR,
                                        fail_reason=str(error_msg))
                    LOG.error("can't go on copy job, set backup %s to error" % backup_id)
                    raise Exception("can't go on copy job")
                LOG.info('verify finished')
            except Exception as err:
                with excutils.save_and_reraise_exception():
                    LOG.error("copy task failed with %s" % err)

    def _get_repository_by_id(self, repository_id):
        get_repository_uri = self.ebackup_manager_repository_uri + \
                             '?id=' + repository_id
        LOG.info('get repository by id:' + repository_id)
        repository = None
        result = self.__get_connection().get(get_repository_uri)
        error_number = self._get_error(result)
        if error_number == ErrorCode.REP_NON_EXISTENT:
            LOG.info('repository does not exist.')
            return repository
        elif error_number != 0:
            msg = self._convert_error(error_number, "query repository info by id is fail")
            LOG.error(msg)
            raise Exception(msg)
        repository = self._get_data(result)
        return repository

    def _update_copy_record(self, backup, metadata, copy_image_id='', volume_backup_size=0):
        LOG.info("begin update copy image info.")
        copy_volume_id = backup['volume_id']
        self._change_value_by_keys(metadata, backupurl_tmp=str(copy_image_id))
        self._remove_keys(metadata, "progress")  # remove progress
        # set volume id
        self.__copy_image_volume_id = str(copy_volume_id)
        backup["service_metadata"] = json.dumps(metadata)
        backup["object_count"] = volume_backup_size

    def run_copy_plan(self, run_copy_body, copy_type):
        run_copy_uri = self.ebackup_copy_plan_uri + '/run'
        for i in range(0, cst.TASK_RETRY_COUNT):
            result = self.__get_connection().put(run_copy_uri, run_copy_body)
            error_number = self._get_error(result)
            if error_number != 0:
                self.error_info.set_error_info(error_number, self._get_error_description(result))
                msg = self._convert_error(error_number, self._get_error_description(result))
                LOG.error(msg)
                raise Exception(msg)

            result_data = self._get_data(result)
            request_id = result_data.get("REQUEST_ID")
            try:
                task_id = self._get_task_id(copy_type, request_id)
                break
            except exception.InvalidBackup as msg:
                if i == (cst.TASK_RETRY_COUNT - 1):
                    LOG.error("task[%s] is not found", request_id)
                    raise exception.InvalidBackup(cst.VERIFY_TASK_NOT_FOUND)
                LOG.warning("get copy task failed:%s, try again %s", msg, i)
        return task_id

    def _query_image_detail_by_id(self, image_id):
        get_image_uri = self.ebackup_image_uri + '?filter=ID:' + image_id
        LOG.info('get image by id:' + image_id)
        result = self.__get_connection().get(get_image_uri)
        error_number = self._get_error(result)
        if error_number != 0:
            self.error_info.set_error_info(error_number, self._get_error_description(result))
            msg = ('query image info by id(%s) is fail' % image_id)
            msg = self._convert_error(error_number, msg)
            LOG.error(msg)
            raise exception.InvalidBackup(msg)
        if "data" not in result:
            msg = ('this image(id:%s) does not exist.' % image_id)
            error_msg = self._convert_error(ErrorCode.IMAGE_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        image_list = result["data"]
        if len(image_list) <= 0:
            msg = ('this image(id:%s) does not exist.' % image_id)
            error_msg = self._convert_error(ErrorCode.IMAGE_NON_EXISTENT, msg)
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        image = image_list[0]
        return image

    @staticmethod
    def _create_copy_policy(conn, policy_name, datalayout):
        rep_body = {
            "NAME": policy_name,
            "DESCRIPTION": "",
            "RETENTIONTYPE": 2,
            "RETENTIONVERS": 0,
            "DATALAYOUT": datalayout,
            "RETRYENABLED": 0,
            "SCHEDULETYPE": POLICY_SCHEDULE_TYPE_ONCE,
            "RENTIONYEARS": 0,
            "RENTIONMONTHS": 0,
            "RENTIONWEEKS": 0,
            "RENTIONDAYS": 0,
            "RETRYINTERVAL": 16,
            "MAXRETRYTIME": 3,
            "ONCEDATE": 1451577600
        }
        result = conn.post("/vcopy_policy", rep_body)
        dict_error = result['error']
        error_code = dict_error['code']
        return error_code

    @staticmethod
    def _update_copy_plan_by_copy_policy(conn, copy_policy_name):
        copy_policy_period_name = str(copy_policy_name) + "_period"
        get_copy_policy_uri = "/vcopy_policy?filter=NAME::" + str(copy_policy_period_name)
        get_copy_policy_result = conn.get(get_copy_policy_uri)
        dict_error = get_copy_policy_result['error']
        error_code = dict_error['code']
        if error_code != 0:
            EbackupDriver.error_info.set_error_info(error_code,
                                                    EbackupDriver._get_error_description(get_copy_policy_result))
            return error_code

        if "data" not in get_copy_policy_result:
            return 0
        copy_policy_list = get_copy_policy_result["data"]
        if len(copy_policy_list) <= 0:
            return 0

        copy_policy = copy_policy_list[0]
        copy_policy_id = copy_policy["ID"]

        modify_copy_plan_body = {"POLICYNAME": copy_policy_name}
        max_num_for_single_query = 100
        while True:
            # get copy plan
            copy_plan_uri = "/vcopy_plan?filter=POLICYNAME::" + str(copy_policy_period_name) + "&range=[0-" + str(
                max_num_for_single_query) + "]"
            get_copy_plan_result = conn.get(copy_plan_uri)
            dict_error = get_copy_plan_result['error']
            error_code = dict_error['code']
            if 0 != error_code:
                EbackupDriver.error_info.set_error_info(error_code, EbackupDriver._get_error_description(dict_error))
                return error_code

            # update copy plan
            if "data" not in get_copy_plan_result:
                break
            copy_plan_list = get_copy_plan_result["data"]
            for copy_plan in copy_plan_list:
                copy_plan_id = copy_plan["ID"]
                modify_copy_plan_uri = "/vcopy_plan/" + str(copy_plan_id)
                modify_copy_plan_result = conn.put(modify_copy_plan_uri, modify_copy_plan_body)
                dict_error = modify_copy_plan_result['error']
                error_code = dict_error['code']
                if 0 != error_code:
                    EbackupDriver.error_info.set_error_info(error_code,
                                                            EbackupDriver._get_error_description(dict_error))
                    return error_code
            if len(copy_plan_list) < max_num_for_single_query:
                break

        # delete period copy policy
        del_copy_policy_uri = "/vcopy_policy/" + str(copy_policy_id)
        del_copy_policy_result = conn.delete(del_copy_policy_uri)
        dict_error = del_copy_policy_result['error']
        error_code = dict_error['code']
        if ErrorCode.COPY_POLICY_NOT_EXISTENT == error_code:
            return 0
        if error_code != 0:
            EbackupDriver.error_info.set_error_info(error_code, EbackupDriver._get_error_description(dict_error))
        return error_code

    @staticmethod
    def _rename_copy_policy_and_create_copy_policy(conn, copy_policy):
        copy_policy_id = copy_policy["ID"]
        copy_policy_name = copy_policy["NAME"]
        copy_policy_period_name = str(copy_policy_name) + "_period"
        modify_copy_policy_body = {"NAME": copy_policy_period_name}
        modify_copy_policy_uri = "/vcopy_policy/" + str(copy_policy_id)
        put_result = conn.put(modify_copy_policy_uri, modify_copy_policy_body)
        dict_error = put_result['error']
        error_code = dict_error['code']
        if ErrorCode.COPY_POLICY_NOT_EXISTENT == error_code:
            return 0
        if 0 != error_code:
            EbackupDriver.error_info.set_error_info(error_code, EbackupDriver._get_error_description(dict_error))
            return error_code

        copy_policy_datalayout = copy_policy["DATALAYOUT"]
        error_code = EbackupDriver._create_copy_policy(conn, copy_policy_name, int(copy_policy_datalayout))
        if ErrorCode.EXISTS_NAME == error_code:
            return 0

        return error_code

    @staticmethod
    def _update_copy_policy(conn, copy_policy):
        copy_policy_schedul_type = copy_policy["SCHEDULETYPE"]
        if str(POLICY_SCHEDULE_TYPE_ONCE) != str(copy_policy_schedul_type):
            error_code = EbackupDriver._rename_copy_policy_and_create_copy_policy(conn, copy_policy)
            if 0 != error_code:
                return error_code

        copy_policy_name = copy_policy["NAME"]
        error_code = EbackupDriver._update_copy_plan_by_copy_policy(conn, copy_policy_name)
        return error_code

    @staticmethod
    def _create_copy_policy_by_name(conn, policy_name, datalayout):
        error_code = EbackupDriver._create_copy_policy(conn, policy_name, datalayout)
        if 0 != error_code and ErrorCode.EXISTS_NAME != error_code:
            return error_code

        error_code = EbackupDriver._update_copy_plan_by_copy_policy(conn, policy_name)
        return error_code

    @staticmethod
    def update_all_copy_policy(ebk_conn):
        # create or update no zip copy policy
        get_copy_policy_uri = "/vcopy_policy?filter=NAME::openstack_default_copy_policy_nozip_noDedupe"
        get_copy_policy_result = ebk_conn.get(get_copy_policy_uri)
        dict_error = get_copy_policy_result['error']
        error_code = dict_error['code']
        if error_code != 0:
            msg = convert_error(error_code, 'Get copy policy failed')
            return -1, msg
        if "data" in get_copy_policy_result:
            copy_policy_list = get_copy_policy_result["data"]
            if len(copy_policy_list) > 0:
                copy_policy = copy_policy_list[0]
                error_code = EbackupDriver._update_copy_policy(ebk_conn, copy_policy)
                if error_code != 0:
                    msg = convert_error(error_code, 'Modify copy policy failed')
                    return -1, msg
        else:
            error_code = EbackupDriver._create_copy_policy_by_name(ebk_conn,
                                                                   "openstack_default_copy_policy_nozip_noDedupe", 0)
            if error_code != 0:
                msg = convert_error(error_code, 'Create copy policy failed')
                return -1, msg
        # create or update zip copy policy
        get_copy_policy_uri = "/vcopy_policy?filter=NAME::openstack_default_copy_policy_zip_noDedupe"
        get_copy_policy_result = ebk_conn.get(get_copy_policy_uri)
        dict_error = get_copy_policy_result['error']
        error_code = dict_error['code']
        if error_code != 0:
            msg = convert_error(error_code, 'Get copy policy failed')
            return -1, msg
        if "data" in get_copy_policy_result:
            copy_policy_list = get_copy_policy_result["data"]
            if len(copy_policy_list) > 0:
                copy_policy = copy_policy_list[0]
                error_code = EbackupDriver._update_copy_policy(ebk_conn, copy_policy)
                if error_code != 0:
                    msg = convert_error(error_code, 'Modify copy policy failed')
                    return -1, msg
        else:
            error_code = EbackupDriver._create_copy_policy_by_name(ebk_conn,
                                                                   "openstack_default_copy_policy_zip_noDedupe", 2)
            if error_code != 0:
                msg = convert_error(error_code, 'Create copy policy failed')
                return -1, msg
        return 0, ""

    def create_or_update_copy_policy(self):
        error_code, error_msg = EbackupDriver.update_all_copy_policy(self.__get_connection())
        if 0 != error_code:
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        LOG.info("create copy policy success.")

    def get_copy_plan_uuid(self, uri, copy_plan_name):
        copy_plan_uuid = None
        result = self._get_object(uri, copy_plan_name)
        error_info = self._get_error(result)
        if error_info != 0:
            LOG.error('get object id failed, eBackup error num is: %d' % error_info)
            return None
        data_info = self._get_data(result)
        if len(data_info) != 0:
            copy_plan_uuid = data_info[0].get("UUID")

        return copy_plan_uuid

    def update_copy_snap_info(self, backup, metadata, copyplan):
        target_repository_uuid, unit_filesystem_type, unit_path, pool_id = self.get_more_info_by_rep_name(
            copyplan.get("TARGETREPOSITORYNAME"))

        def degrade_layout(layout):
            # if layout has dedupe, we should use nodedupe
            if layout is None or not layout.isdigit():
                return layout

            if int(layout) > 1:
                return 2
            else:
                return 0

        if metadata.get('IT') == cst.IMPORT_TYPE_COPY:
            if "nozip" in copyplan.get("POLICYNAME"):
                data_layout = 0
            else:
                data_layout = 2
        else:
            data_layout = degrade_layout(str(metadata.get('DL')))
        self._change_value_by_keys(metadata,
                                   BT=unit_filesystem_type,
                                   BP=unit_path,
                                   DL=data_layout)
        backup["service_metadata"] = Utils.transfer_to_json_str(metadata)

    def recove_backupurl(self, backup_id, task_type):
        backup = call_db_fun(self.db.backup_get, self.context, backup_id)
        LOG.info("update cinder db import type with task type[%d]." % task_type)
        metadata = json.loads(backup['service_metadata'])
        self._remove_keys(metadata, *REMOTE_REPLICATION_POP_KEY)
        if task_type == cst.TASKTYPE_DIRECT_RESTORE:
            self._change_value_by_keys(metadata, IT=cst.IMPORT_TYPE_DIFFERENT_AZ_RESTORE,
                                       backupurl=metadata.get("backupurl_tmp"))
        elif task_type == cst.TASKTYPE_NATIVE_TO_CLOUD:
            self._change_value_by_keys(metadata, IT=cst.IMPORT_TYPE_NATIVE_TO_CLOUD,
                                       backupurl=metadata.get("backupurl_tmp"))
        self._remove_keys(metadata, "backupurl_tmp")
        self._update_backup(backup_id, service_metadata=json.dumps(metadata), object_count=-1)

    def encrypt_data(self, data):
        body = {"CONTENT": str(data)}
        rsp = self.__get_connection().put(self.ebackup_encrypt_url, body, sync=True)
        err_code = self._get_error(rsp)
        if err_code != 0:
            LOG.error("encrypt_data failed")
            return "", False
        data = self._get_data(rsp)
        if "CONTENT" in data and data.get("CONTENT"):
            return data.get("CONTENT"), True
        else:
            LOG.error("encrypt_data is empty")
            return "", False

    def check_export_data_format(self, meta):
        if not meta:
            return False
        check_null = lambda x: x == ""
        check_ipv4 = lambda x: Utils.check_string(x, cst.ip_v4_compile)
        check_ipv6 = lambda x: Utils.check_string(x, cst.ip_v6_compile)
        need_check_str_key_list = ['SP', 'DL', 'ST', 'BT', 'BP', 'EBKPORT', 'AK', 'SK', 'OBPID', 'PR', 'SAZ']
        need_check_ip_key_list = ['EBKIP']
        for key in need_check_str_key_list:
            if check_null(meta.get(key, "")):
                return False
        for key in need_check_ip_key_list:
            if not check_ipv4(meta.get(key, "")) and not check_ipv6(meta.get(key, "")):
                return False
        return True

    def export_encrypt_info(self, backup, metadata):
        try:
            dek_encrypt_ok = True
            evs = EVS()
            evs_body = evs.create_evs_body(self.db, self.context, backup, False)
            if evs_body['DATA_ENC_INFO']['CMKID']:
                encrypt_dek, dek_encrypt_ok = self.encrypt_data(evs_body['DATA_ENC_INFO']['DEK'])
                self._change_value_by_keys(metadata, DEK=encrypt_dek)
                if not metadata.get('CMKID') or not metadata.get('VK'):
                    self._change_value_by_keys(metadata,
                                               CMKID=evs_body['DATA_ENC_INFO']['CMKID'],
                                               VK=evs_body['DATA_ENC_INFO']['VK'])
        except Exception as msg:
            LOG.warning("get encypt data failed(reason: %s), check the last backup of volume:%s", msg,
                        backup['volume_id'])
            backup_list = call_db_fun(self.db.backup_get_by_volume_id, self.context, backup['volume_id'])
            last_backup = self.__get_last_backup(backup_list, False, "", "copy")
            dek_encrypt_ok = True
            if last_backup:
                l_backup_meta = Utils.transfer_json_to_dict(last_backup.get('service_metadata'))
                if l_backup_meta.get("CMKID") and l_backup_meta.get("VK"):
                    evs = EVS()
                    evs.decrypt_evs_body(self.db, self.context, backup, False)
                    encrypt_dek, dek_encrypt_ok = self.encrypt_data(evs.DEK)
                    self._change_value_by_keys(metadata, DEK=encrypt_dek)
        return dek_encrypt_ok

    @profiler.trace("ebackup driver:invoke to export_record")
    @HandleErrorDesc(ErrorCode.BD_EXPORT_RECORD_FAILED)
    def export_record(self, backup):
        try:
            self.error_info.error_code = ErrorCode.BD_EXPORT_RECORD_FAILED
            self.__task_type = 'export'
            self.__backup = backup
            global ebackup_driver_conf
            ebackup_driver_conf = backup_config()
            ebackup_driver_conf.init_conf()
            self.__availability_zone = backup['availability_zone']
            json_metadata = backup.get('service_metadata')
            metadata = json.loads(json_metadata)
            self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
            # before assign, you should assign value for __availability_zone and __physical_network_type
            ebk_client = EbackupConnectionClient(self.__get_connection())
            # export DL attribute
            dlAttribute = metadata.get('DL')
            if dlAttribute in (None, ''):
                LOG.info('export DL for backup')
                self.update_database_for_backup(backup)
                backup_id = backup.get('id')
                newbackup = call_db_fun(self.db.backup_get, self.context, backup_id)
                json_metadata = newbackup.get('service_metadata')
                metadata = json.loads(json_metadata)
            backup_url = metadata.get('backupurl')
            self._remove_keys(metadata, "backupurl", "OMID", "OMIP", "DT", "VT", "AT", "CS", "CT", "SS", "AD", "B_ID")
            self._change_value_by_keys(metadata, backupurl_tmp=str(backup_url), ebk_T_I='')
            # export dek and ebk info for remote replication
            if ebackup_driver_conf.ebackup_env_type == ENV_PUBLIC_CLOUD:
                dek_encrypt_ok = self.export_encrypt_info(backup, metadata)
                ebackup_server_ip = self.__get_connection().get_ip()
                ebackup_server_port = self.__get_connection().get_port()
                if not ebackup_driver_conf.ebackup_storage_username or not ebackup_driver_conf.ebackup_storage_password:
                    LOG.error("ak or sk is null")
                    raise exception.InvalidBackup("ak or sk is null")
                storage_username = Utils.decrypt_password(ebackup_driver_conf.ebackup_storage_username)
                storage_password = Utils.decrypt_password(ebackup_driver_conf.ebackup_storage_password)
                if not storage_username or not storage_password:
                    LOG.error("decrypted ak or sk is null")
                    raise exception.InvalidBackup("decrypted ak or sk is null")
                encrypt_sk, sk_encrypt_ok = self.encrypt_data(storage_password)
                backup_plan_uuid = metadata.get('SP').split('/')[1]
                self._change_value_by_keys(metadata, EBKIP=ebackup_server_ip, EBKPORT=ebackup_server_port,
                                           AK=storage_username,
                                           SK=encrypt_sk, OBPID=backup_plan_uuid, ODL=metadata.get('DL'),
                                           PR=ebackup_driver_conf.ebackup_storage_s3_protocol,
                                           SAZ=backup['availability_zone'])

                check_result = self.check_export_data_format(metadata)
                if not check_result or not dek_encrypt_ok or not sk_encrypt_ok:
                    backup['fail_reason'] = self._convert_error(ErrorCode.BD_EXPORT_RECORD_FAILED,
                                                                "BD_EXPORT_RECORD_FAILED")
                else:
                    backup['fail_reason'] = ""
            elif ebackup_driver_conf.ebackup_env_type == ENV_PRIVATE_CLOUD:
                # private cloud not support encrypt backup
                ebackup_server_ip = self.__get_connection().get_ip()
                ebackup_server_port = self.__get_connection().get_port()
                # private cloud should get as/sk from ebackup server
                storage_units, err, desc = \
                    ebk_client.get_storage_unit_by_image_id(
                        backup_url, metadata.get('BP', ''))
                if err == 0:
                    ak_p = storage_units.get("USERNAME", "")
                    sk_p = storage_units.get("PASSWORD", "")
                    nas_path = storage_units.get("PATH", "")
                    self._change_value_by_keys(metadata, AK=ak_p,
                                               SK=sk_p, NPATH=nas_path)
                else:
                    LOG.error("not found any storage unit with export!,%s,%s", err, desc)
                backup_plan_uuid = metadata.get('SP').split('/')[1]
                self._change_value_by_keys(metadata, EBKIP=ebackup_server_ip, EBKPORT=ebackup_server_port,
                                           OBPID=backup_plan_uuid, ODL=metadata.get('DL'),
                                           SAZ=backup['availability_zone'])
                check_result = self.check_export_data_format(metadata)
                if not check_result:
                    backup['fail_reason'] = self._convert_error(ErrorCode.BD_EXPORT_RECORD_FAILED,
                                                                "BD_EXPORT_RECORD_FAILED")
                else:
                    backup['fail_reason'] = ""
            backup['service_metadata'] = Utils.transfer_to_json_str(metadata)
            result = super(EbackupDriver, self).export_record(backup)
            if not result:
                driver_info = {}
                driver_info['backup_info'] = backup.encode_record()
                return driver_info
            else:
                return result
        except Exception as msg:
            LOG.error(msg)
            LOG.error('Origin stack is:%s' % traceback.format_exc())
            raise exception.InvalidBackup(msg)

    def update_copy_flag(self, backup, metadata, import_type=cst.DOMAIN_TYPE_COPY):
        LOG.info("begin update copy flag info.")

        try:
            st = metadata["ST"]
        except:
            st = None

        if st in (None, 0):
            st = cst.COPY_TYPE_SINGLE_VOL
        elif st in (1,):
            st = cst.COPY_TYPE_MACHINE
        elif st in (4, 5):
            pass
        else:
            LOG.info("ST should be 0 or 1, and now ST is " + str(st))
            st = cst.COPY_TYPE_SINGLE_VOL

        self._change_value_by_keys(metadata, ST=st, IT=import_type)
        backup["service_metadata"] = Utils.transfer_to_json_str(metadata)

    def _do_import_record(self, backup, backup_id):
        """The internal implement of copy image for M and J version."""
        tenant_id = backup['project_id']
        self.__availability_zone = backup['availability_zone']

        if self._check_length(tenant_id, cst.LENGTH_LIMIT_255):
            msg = ('tenant id length does not comply, error length:%d' % len(tenant_id))
            error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
            LOG.error(error_msg)
            raise Exception(error_msg)
        LOG.info('Begin hwsebackup copy, tenant id:%s' % tenant_id)

        metadata = json.loads(backup['service_metadata'])
        # fetch pnt value for get_connection
        self.__physical_network_type = 'ib' if metadata.get('IB', 0) == 1 else 'ip'
        LOG.info("physical network type: %s", self.__physical_network_type)
        backup_image_id = metadata['backupurl_tmp']
        self.__copy_image_source_id = backup_image_id
        copy_volume_id = backup['volume_id']

        self.update_copy_flag(backup, metadata)
        self._remove_keys(metadata, *REMOTE_REPLICATION_POP_KEY)
        self._update_backup(backup_id, service_metadata=json.dumps(metadata), created_at=timeutils.utcnow())
        self.update_om_hostid_for_backup("", backup_id, True)

        # check whether copied
        copy_snap_guid_list, service_metadata = self._get_copy_snap_guids_by_snap_id(backup_image_id,
                                                                                     str(copy_volume_id), True)
        if len(copy_snap_guid_list) != 0:
            LOG.info("this backup image has been copied")
            self.__is_image_has_copied = True
            self._change_value_by_keys(metadata,
                                       BT=service_metadata.get("BT"),
                                       BP=service_metadata.get("BP"),
                                       DL=service_metadata.get("DL"),
                                       SP=service_metadata.get("SP"),
                                       backupurl=copy_snap_guid_list[0])
            self._update_backup(backup_id, service_metadata=Utils.transfer_to_json_str(metadata))
            return backup

        ebk_T_I = metadata.get('ebk_T_I', '')
        if ebk_T_I.startswith('R_'):
            request_id = ebk_T_I[2:]
            try:
                task_id = self._get_task_id(cst.TASKTYPE_COPY, request_id)
                if task_id:
                    LOG.info("Get ebk_T_I {ebk_T_I} by request_id".format(
                        ebk_T_I=task_id))
                    self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
                    self._update_backup(
                        backup_id, service_metadata=json.dumps(metadata))
                    return
            except Exception:
                LOG.info("Failed to get ebk_T_I by request_id {request_id}".
                         format(request_id=request_id))
        else:
            if ebk_T_I:
                LOG.info("Get ebk_T_I {ebk_T_I} from metadata".format(
                    ebk_T_I=ebk_T_I))
                return

        # get source repository name by backup image id
        backup_image = self._get_image_by_id(backup_image_id)
        if backup_image is None:
            msg = self._convert_error(ErrorCode.IMAGE_NON_EXISTENT, "this backup image does not exist")
            LOG.error(msg)
            raise Exception(msg)
        src_repository_id = backup_image['ASSOCIATENAMESPACEID']
        src_repository = self._get_repository_by_id(src_repository_id)
        if src_repository is None:
            msg = self._convert_error(ErrorCode.REP_NON_EXISTENT, "this repository does not exist.")
            LOG.error(msg)
            raise Exception(msg)
        src_repository_name = src_repository['NAME']

        # check the src storage unit whether is S3
        src_spool_id = src_repository['PARENTID']
        filter_string = "?filter=parentid::" + str(src_spool_id)
        get_storage_unit_ret = self._get_object_by_filter(
            self.ebackup_manager_unit_list_uri, filter_string)
        error_code = self._get_error(get_storage_unit_ret)
        if 0 != error_code:
            msg = self._convert_error(error_code, "get storage unit infomation failed.")
            LOG.error(msg)
            raise Exception(msg)
        src_storage_unit = self._get_data(get_storage_unit_ret)
        if src_storage_unit is None:
            msg = self._convert_error(ErrorCode.BRICK_NON_EXISTENT, "the storage unit does not exist.")
            LOG.error(msg)
            raise Exception(msg)
        if len(src_storage_unit) == 0:
            msg = self._convert_error(ErrorCode.BRICK_NON_EXISTENT, "the storage unit does not exist.")
            LOG.error(msg)
            raise Exception(msg)

        # create or update copy policy
        self.create_or_update_copy_policy()
        # create copy plan, repository, storage pool
        copy_plan_name = self._create_copy_related_resource(backup, src_repository_name)

        # if sourcerepository in copyplan is none, then update it in copyplan
        self._update_hybrid_related_resource(backup, src_repository_name)

        # get copy plan ID
        ebk_client = EbackupConnectionClient(self.__get_connection())
        copy_plan_list, error_code, error_msg = ebk_client.get_copy_plan_by_name(copy_plan_name)
        if error_code != 0 or not copy_plan_list:
            msg = self._convert_error(ErrorCode.COPY_PLAN_NON_EXISTENT, "this copy plan does not exist.")
            LOG.error(msg)
            raise Exception(msg)
        copy_plan = copy_plan_list[0]
        copy_plan_id = copy_plan['ID']
        if self.__version_info == 'M':
            self.update_copy_snap_info(backup, metadata, copy_plan)
            self._update_backup(backup_id, service_metadata=backup['service_metadata'])
        # copy
        self.__backup_orig_id = backup_id
        request_id = Utils.gen_request_id()
        run_copy_body = {
            "ID": copy_plan_id,
            "TYPE": COPY_PLAN_OBJ_ID,
            "BACKUPIMAGEID": backup_image_id,
            "REQUEST_ID": request_id
        }
        metadata = Utils.transfer_json_to_dict(backup['service_metadata'])
        req_id = 'R_' + request_id
        self._change_value_by_keys(metadata, ebk_T_I=str(req_id))
        self._update_copy_record(backup, metadata, backup_image_id)
        # save req id before run copy plan
        self._update_backup(backup_id, service_metadata=json.dumps(metadata))

        LOG.info("the copy task start running, req_id:{}.".format(req_id))
        task_id = self.run_copy_plan(run_copy_body, cst.TASKTYPE_COPY)
        self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
        self._update_copy_record(backup, metadata, backup_image_id)
        self._update_backup(backup_id, service_metadata=json.dumps(metadata))
        return backup

    @profiler.trace("ebackup driver:invoke to import_recod")
    @HandleErrorDesc(ErrorCode.BD_IMPORT_RECORD_FAILED)
    def import_record(self, backup, *args):
        """Implement the copy single backup image for M/J version, M version import_type field is in driver_info."""
        msg = "enter the import special backup image."
        LOG.info(msg)
        self.error_info.error_code = ErrorCode.BD_IMPORT_RECORD_FAILED
        # Init driver config
        global ebackup_driver_conf
        ebackup_driver_conf = backup_config()
        ebackup_driver_conf.init_conf()
        try:
            if len(args) > 0:
                # M
                driver_info = args[0]

                if "import_type" not in driver_info:
                    # call native interface
                    LOG.info("execute the native function of import record.")
                    return super(EbackupDriver, self).import_record(backup, driver_info)

                # check whether support the import type
                import_type = int(driver_info['import_type'])
                import_dict = {cst.IMPORT_TYPE_COPY: cst.TASKTYPE_COPY,
                               cst.IMPORT_TYPE_DIFFERENT_AZ_RESTORE: cst.TASKTYPE_DIRECT_RESTORE,
                               cst.IMPORT_TYPE_NATIVE_TO_CLOUD: cst.TASKTYPE_NATIVE_TO_CLOUD,
                               cst.IMPORT_TYPE_REMOTE_REPLICATION: cst.TASKTYPE_REMOTE_REPLICATION,
                               cst.IMPORT_TYPE_REMOTE_RESTORE: cst.TASKTYPE_VOL_REMOTE_RESTORE}
                self.__driver_import_type = import_type
                self.__driver_task_type = import_dict.get(import_type)
                if self.__driver_task_type is None:
                    msg = ("the import type can not support, import type is:%d" % import_type)
                    error_msg = convert_error(ErrorCode.BD_IMPORT_TYPE_NOT_SUPPORT, msg)
                    LOG.error(error_msg)
                    raise Exception(error_msg)

                if "backup_info" not in driver_info:
                    LOG.info("execute the native function of import record, backup_info is not in driver_info.")
                    return super(EbackupDriver, self).import_record(backup, driver_info)

                backup_url = driver_info['backup_info']
                backup = objects.Backup.decode_record(backup_url)

                self.__version_info = 'M'
                self.__backup_info = backup
                self.__copy_image_volume_id = str(backup['volume_id'])
                import_result = backup
            else:
                # J
                backup_url = backup
                self.__driver_task_type = cst.TASKTYPE_COPY
                # check backup_url length
                MAX_BACKUP_URL_LENGTH = 10240
                if len(str(backup_url)) > MAX_BACKUP_URL_LENGTH:
                    msg = "the backup url is too long."
                    error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                    LOG.error(error_msg)
                    raise Exception(error_msg)

                backup = json.loads(backup_url.decode("base64"))
                if "import_type" not in backup:
                    # call native interface
                    LOG.info("execute the native function of import record.")
                    return super(EbackupDriver, self).import_record(backup_url)

                # check whether support the import type
                import_type = int(backup['import_type'])

                # check whether support the import type
                if import_type != cst.IMPORT_TYPE_COPY:
                    msg = ("the import type can not support, import type is:%d" % import_type)
                    error_msg = convert_error(ErrorCode.BD_IMPORT_TYPE_NOT_SUPPORT, msg)
                    LOG.error(error_msg)
                    raise Exception(error_msg)

                self.__version_info = 'J'
                self.__backup_info = backup
                self.__copy_image_volume_id = str(backup['volume_id'])
                import_result = backup
        except Exception as msg:
            LOG.error("import error:%s" % str(msg))
            LOG.error('Origin stack is:%s' % traceback.format_exc())
            raise exception.InvalidBackup(str(msg))
        return import_result

    def redo_backup(self, backup):
        """Backup the given volume to eBackup.

        eBackup performs a backup of a volume. The volume_file is used
        to determine the path of the block device that eBackup will back-up.

        :param backup: backup information for volume
        :raises InvalidBackup
        """
        try:
            # get ebk_T_I
            backup_id = backup['id']
            backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            json_metadata = backup.get('service_metadata')
            if json_metadata not in ("", None):
                metadata = json.loads(json_metadata)
            self.init_evs_body(backup, "backup")
            request_id = metadata.get('ebk_T_I')
            LOG.info('the request_id is: %s' % request_id)
            backup[cst.LLD_DISABLED] = True
            if request_id is None:
                LOG.info('the metadata progress is: %s' % metadata.get('progress'))
                # ebk_T_I is none
                if metadata.get('progress', '0') == '1':
                    # check whether has plan,if not create it
                    self._check_storage_ralate(backup)
                    self.update_database_for_backup(backup)
                    # in this case backup.volume_id is updated to source_volume_id

                    # M version snapshot backup (backup has snapshot_id and it's not none)
                    if ('snapshot_id' in backup.keys()) and (backup.get('snapshot_id') is not None):
                        LOG.info("Snapshot backup in M version.")
                        volume_file = self.__get_backup_snapshot_volume_file(backup)
                    # M version link clone volume backup
                    else:
                        LOG.info("Link clone volume backup in M version.")
                        volume_file = self.__get_redo_backup_volume_file(backup)

                    self._run_backup_plan(backup, volume_file)
                    return 1
                    # no need loop to query the progress
                else:
                    # process != 1 do backup,it mains the volume_id is not updated to source_volume_id
                    LOG.info('begin to call native interface backup()')
                    device_file = None
                    self.backup(backup, device_file, backup_metadata=False)
                    return 1  # no need to loop query the progress
            else:
                # ebk_T_I has value
                LOG.info('the request_id is not None')
                request_id_tmp = None  # the truely request_id
                # check whether have task
                if request_id.startswith('R_'):
                    request_id_tmp = request_id[2:]
                    task_id = self._get_task_id(cst.TASKTYPE_BACKUP, request_id_tmp, self._get_cascaded_volume_id(backup))
                else:
                    task_id = request_id
                if task_id in (None, ''):
                    # retribute run_plan_body
                    LOG.info('task_id is not exist')
                    # we could create new backup plan and run it in this case
                    self._check_storage_ralate(backup)
                    self.update_database_for_backup(backup)
                    # M version snapshot backup (backup has snapshot_id and it's not none)
                    if ('snapshot_id' in backup.keys()) and (backup.get('snapshot_id') is not None):
                        LOG.info("Snapshot backup in M version.")
                        volume_file = self.__get_backup_snapshot_volume_file(backup)
                    # M version link clone volume backup
                    else:
                        LOG.info("Link clone volume backup in M version.")
                        volume_file = self.__get_redo_backup_volume_file(backup)

                    self._run_backup_plan(backup, volume_file, request_id_tmp)
                    LOG.info('hwsebackup redo backup run backup plan end.')
                    return 1
                else:
                    # loop to query the progress
                    LOG.info('task is exsit, no need to create task_id is: %s ' % task_id)
                    task_infos = self.get_task_info(task_id)
                    real_plan_name = task_infos.get('ASSOCIATEBACKUPDOMAINNAME')
                    LOG.info("task's redo plan name : %s", real_plan_name)
                    backup[cst.PLAN_NAME] = real_plan_name
                    self.update_database_for_backup(backup)

        except Exception as msg:
            LOG.error('hwsebackup redo backup failed %s!' % msg)
            raise exception.InvalidBackup(msg)

        LOG.info('hwsebackup redo backup end.')
        return 0  # loop to query the progress


    def redo_restore(self, backup, step):
        """Backup the given volume to eBackup.

        eBackup performs a backup of a volume. The volume_file is used
        to determine the path of the block device that eBackup will back-up.

        :param backup: backup information for volume
        :raises InvalidBackup
        """
        try:
            # get ebk_T_I
            json_metadata = backup.get('service_metadata')
            metadata = json.loads(json_metadata)
            backup_id = backup['id']
            request_id = metadata.get("ebk_T_I")
            target_volume_id = backup.restore_volume_id
            LOG.info('restore target volume id : %s.' % target_volume_id)
            self.init_evs_body(backup, 'restore')
            if step <= cst.STEP_LLD_FAILED:  # request_id is None:
                # ebk_T_I is none ,generate request_id,then run restore
                LOG.info('ebk_T_I is none.')
                self.restore(backup, target_volume_id, None)
                self._update_backup(backup_id, status=cst.BACKUP_STATUS_AVALIBLE)
                call_db_fun(self.db.volume_update, self.context, target_volume_id, {'status': 'available'})

            else:
                # ebk_T_I has value
                # check whether have task
                LOG.info('the request_id is not None')
                task_type_restore = cst.TASKTYPE_VOL_RESTORE
                if metadata.get("C-Region"):
                    task_type_restore = cst.TASKTYPE_VOL_REMOTE_RESTORE
                if step == cst.STEP_HAS_REQUEST_STARTWITH_R:  # request_id.startswith('R_'):
                    request_id_tmp = request_id[2:]
                    task_id = self._get_task_id(task_type_restore, request_id_tmp)
                else:
                    task_id = request_id

                # TODO:
                if task_id is not None:
                    LOG.info('restore task id is: %s' % task_id)
                    task_infos = self.get_task_info(task_id)
                    if not isinstance(task_infos, dict):
                        LOG.info('restore task not exsit.')
                        self.restore(backup, target_volume_id, None)
                        self._update_backup(backup_id, status=cst.BACKUP_STATUS_AVALIBLE)
                        call_db_fun(self.db.volume_update, self.context, target_volume_id, {'status': 'available'})
                        return

                    task_status = task_infos.get("STATUS")
                    LOG.info('restore task status is: %s' % task_status)
                    if task_status == cst.TASK_STATUS_COMPLETED:
                        LOG.info('restore task exsit and complete.')
                        self.restore(backup, target_volume_id, None)
                        self._update_backup(backup_id, status=cst.BACKUP_STATUS_AVALIBLE)
                        call_db_fun(self.db.volume_update, self.context, target_volume_id, {'status': 'available'})
                    else:
                        # loop to query the progress
                        LOG.info('begin loop to query the restoring progress.')
                        self.query_redo_restore_process(backup, task_id, task_type_restore)
                        LOG.info('end query_restore_backup_process.')
                        pass
                else:
                    # has no task
                    LOG.info('restore task not exsit.')
                    self.restore(backup, target_volume_id, None)
                    self._update_backup(backup_id, status=cst.BACKUP_STATUS_AVALIBLE)
                    call_db_fun(self.db.volume_update, self.context, target_volume_id, {'status': 'available'})

        except Exception as msg:
            LOG.error('redo restore failed %s' % msg)
            raise exception.InvalidBackup(msg)

        LOG.info(' hwsebackup redo restore end.')
        return 0

    def set_fusionSphere_ip_while_verify(self, backup_status, backup):
        if is_infiniBand_env:
            try:
                fusion_storage_url = ''
                if backup_status == cst.BACKUP_STATUS_CREATING:
                    source_vol = call_db_fun(self.db.volume_get, self.context, backup['volume_id'])
                    volume_storage_type = self.__get_volume_storage_type(source_vol)
                    if cst.STORAGE_TYPE_VRM_FUSION_STORAGE == volume_storage_type \
                            or cst.STORAGE_TYPE_KVM_FUSION_STORAGE == volume_storage_type \
                            or cst.STORAGE_TYPE_VRM_FUSION_RDM == volume_storage_type:
                        fusion_storage_url = self.__get_snapshot_url(volume_storage_type, source_vol,
                                                                     backup['snapshot_id'], False)
                elif backup_status == cst.BACKUP_STATUS_RESTORING:
                    target_volume_id = backup.restore_volume_id
                    volume_info = call_db_fun(self.db.volume_get, self.context,
                                              target_volume_id)  # get the target volume info
                    volume_storage_type = self.__get_volume_storage_type(volume_info)
                    if cst.STORAGE_TYPE_VRM_FUSION_STORAGE == volume_storage_type \
                            or cst.STORAGE_TYPE_KVM_FUSION_STORAGE == volume_storage_type \
                            or cst.STORAGE_TYPE_VRM_FUSION_RDM == volume_storage_type:
                        fusion_storage_url = self.__get_fusionstorage_volume_url(volume_info, volume_storage_type)
                if fusion_storage_url:
                    fusionstorage_info = fusion_storage_url.split("/")
                    if len(fusionstorage_info) < 3:
                        LOG.info("get fusionSphere ip failed, set __fusionstorage_ip failed")
                        return
                    self.__fusionstorage_ip = fusionstorage_info[2]
                LOG.info("fusion storage ip:%s", self.__fusionstorage_ip)
            except Exception as msg:
                LOG.error(msg)
                raise exception.InvalidBackup(msg)

    def lld_restore_step(self, metadata):
        task_id = metadata.get("ebk_T_I", '')
        step_dict = {cst.TAG_LLD_START: cst.STEP_LLD_START,
                     cst.TAG_LLD_SUCCESS: cst.STEP_LLD_SUCCESS,
                     cst.TAG_LLD_FAILED: cst.STEP_LLD_FAILED}
        if task_id:
            step = step_dict.get(task_id, None)
            if step is None:
                step = cst.STEP_HAS_REQUEST_STARTWITH_R if task_id.startswith('R_') else cst.STEP_HAS_SERVER_REQUEST
        else:
            step = cst.STEP_LLD_START
        LOG.info("restore step: %d", step)
        return step

    def do_lld_restore_after_interrupt(self, redobackup, om_host_id, current_host_id, step):
        need_update_vol_and_backup_status = False
        lld_success = False
        if om_host_id and om_host_id == current_host_id:
            if cst.STEP_LLD_SUCCESS == step:
                LOG.info("end backup[%s] redo restore:lld restore have been finished", redobackup['id'])
                need_update_vol_and_backup_status = True
                lld_success = True
            if cst.STEP_LLD_START == step:
                is_done = self.do_lld_restore(redobackup, redobackup.restore_volume_id)
                if is_done:
                    LOG.info("end backup[%s] redo restore:lld restore finished", redobackup['id'])
                    need_update_vol_and_backup_status = True
                    lld_success = True
                else:
                    LOG.info("end backup[%s] redo restore: enter original restore flow since lld failed" %
                             redobackup['id'])
        elif om_host_id and om_host_id != current_host_id:
            LOG.info("end backup[%s] redo restore: check it at host[%s]", redobackup['id'], om_host_id)
            lld_success = True

        if need_update_vol_and_backup_status:
            backup_in_database = call_db_fun(self.db.backup_get, self.context, redobackup['id'])
            json_service_metadata = backup_in_database.get('service_metadata')
            update_metadata = Utils.transfer_json_to_dict(json_service_metadata)
            # update task id to empty prevent for not do lld in verify
            self._change_value_by_keys(update_metadata, ebk_T_I="")
            self._update_backup(redobackup['id'], status=cst.BACKUP_STATUS_AVALIBLE,
                                service_metadata=json.dumps(update_metadata))
            call_db_fun(self.db.volume_update, self.context, redobackup.restore_volume_id, {'status': 'available'})

        return lld_success

    def arbitration_node(self, metadata, backup_id):
        om_host_id = metadata.get("OMID", "")
        if om_host_id == "":
            node_uuid = self.update_om_hostid_for_backup("", backup_id)
            LOG.info('set om_host_id to [%s]', node_uuid)
            time.sleep(cst.ALL_NODE_UPDATE_WITH_REDO)
            redobackup = call_db_fun(self.db.backup_get, self.context, backup_id)
            metadata = self.__get_backup_metadata(redobackup)
            om_host_id = metadata.get("OMID", "")
        else:
            node_uuid = Utils.get_hostname()

        if om_host_id != node_uuid:
            LOG.info('the om_host_id is [%s].dont redo on this node[%s]', om_host_id, node_uuid)
            return False
        else:
            LOG.info('start continue backup[%s] on this node[%s]', backup_id, node_uuid)
            return True

    def verify_for_M(self, backup):
        """verify the given backup from eBackup server.
        :param backup: backup id in cinder db
        :raises InvalidBackup
        """
        # redo
        redobackup = backup
        volumeid = redobackup["volume_id"]
        backup_status = redobackup.get('status')
        self.__availability_zone = redobackup.get('availability_zone')
        ''' start check OMID exist
            if om_host_id is none.we need choose a node to continue this task.
        '''
        metadata = self.__get_backup_metadata(redobackup)
        om_host_id = metadata.get("OMID", "")
        self.set_fusionSphere_ip_while_verify(backup_status, redobackup)
        cashier = ExpensesEvent(self.db, 'backup')
        try:
            if backup_status == cst.BACKUP_STATUS_CREATING:
                LOG.info('the redobackup status is creating.')
                self.__task_type = 'backup'
                volume = call_db_fun(self.db.volume_get, self.context, volumeid)
                #availability_zone not contain in redobackup
                self.__availability_zone = volume['availability_zone']
                self._update_backup(backup['id'], availability_zone=self.__availability_zone)
                LOG.info('Redo backup %s volume %s AZ %s' % (redobackup["volume_id"], volumeid, self.__availability_zone))
                redobackup = call_db_fun(self.db.backup_get, self.context, backup['id'])
                self.__backup = redobackup
                if self._get_cloud_service_type(backup) == 'CBR':
                    self.backup(backup, None, False, verify=True)
                else:
                    # redo bakcup
                    LOG.info('begin redo backup.')
                    redobackup[cst.LLD_DISABLED] = True
                    if 1 == self.redo_backup(redobackup):
                        # call backup(),no need to query the progress
                        self._update_backup(backup['id'], status=cst.BACKUP_STATUS_AVALIBLE,
                                            service='cinder.backup.drivers.hwsebackup')
                        LOG.info("The volume_file [%s] is constructed", volumeid)
                        if ('snapshot_id' in redobackup.keys()) and (redobackup.get('snapshot_id') is not None):
                            call_db_fun(self.db.snapshot_update, self.context, redobackup.get('snapshot_id'),
                                        {'status': 'available'})
                        else:
                            call_db_fun(self.db.volume_update, self.context, volumeid, {'status': 'available'})
                        LOG.info('redo backup {backup_id} success.'.format(
                            backup_id=backup['id']))
                    else:
                        LOG.info('begin to query_redo_backup_process.')
                        self.query_redo_backup_process(redobackup)
                        LOG.info('query_redo_backup_process success.')
                redobackup = call_db_fun(self.db.backup_get, self.context, backup['id'])
                cashier.send_expenses_event(self.context, redobackup, redobackup, successd=True)
        except Exception as msg:
            with excutils.save_and_reraise_exception():
                LOG.error("redo backup error(%s) happened", msg)
                self._update_backup(backup['id'],
                                    status=cst.BACKUP_STATUS_ERROR,
                                    service='cinder.backup.drivers.hwsebackup')
                if ('snapshot_id' in redobackup.keys()) and (redobackup.get('snapshot_id') is not None):
                    call_db_fun(self.db.snapshot_update, self.context, redobackup.get('snapshot_id'),
                                {'status': 'available'})
                else:
                    call_db_fun(self.db.volume_update, self.context, volumeid, {'status': 'available'})
                redobackup = call_db_fun(self.db.backup_get, self.context, backup['id'])
                cashier.send_expenses_event(self.context, redobackup, redobackup, successd=False)

        try:
            if backup_status == cst.BACKUP_STATUS_RESTORING:
                LOG.info('the backup status is restoring.')
                self.__task_type = 'restore'
                self.__backup = redobackup
                # redo restore through lld
                step = self.lld_restore_step(metadata)
                current_host_id = Utils.get_hostname()

                if self.do_lld_restore_after_interrupt(redobackup, om_host_id, current_host_id, step):
                    return

                self.not_do_lld_in_redo_restore = True

                # redo restore on ebackup server
                self.redo_restore(redobackup, step)
                LOG.info('redo restore success.')
                return
        except Exception as msg:
            LOG.error(msg)
            LOG.info("redo restoring error(%s) happened", msg)
            self._update_backup(backup['id'], status=cst.BACKUP_STATUS_AVALIBLE)
            call_db_fun(self.db.volume_update, self.context, redobackup.restore_volume_id,
                        {'status': 'error_restoring'})
            return

    def __get_redo_backup_volume_file(self, backup):
        """Get volume file object representing the volume if not supply this information."""

        volume_file = {}
        vol_info = call_db_fun(self.db.volume_get, self.context, backup['volume_id'])
        LOG.info('a backup for volume %s, id : %s.' % (vol_info['name'], backup['volume_id']))

        # Get AZ name
        self.__availability_zone = vol_info['availability_zone']

        # Get storage type.
        volume_storage_type = self.__get_volume_storage_type(vol_info)
        volume_file['storage_type'] = int(volume_storage_type)

        # Get snapshot_id
        service_metadata = self.__get_backup_metadata(backup)
        if not service_metadata:
            error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "backup service_metadata is none.")
            LOG.error(error_msg)
            raise exception.InvalidBackup(error_msg)
        snapshot_id = service_metadata.get('snap_id')
        source_snapshot = call_db_fun(self.db.snapshot_get, self.context, snapshot_id)
        source_volume_id = source_snapshot['volume_id']

        backup_list = self.__backup_get_by_volume_id(source_volume_id)
        last_backup = self.__get_last_backup(backup_list, False, "", "backup")
        if cst.STORAGE_TYPE_KVMV3 == volume_storage_type:
            vol_info = call_db_fun(self.db.volume_get, self.context, source_volume_id)

        # Get backup mode from inc
        if self._get_backup_mode_from_inc(backup) is cst.BACKUP_CMD_FULL:
            # do full backup
            LOG.info('Do full backup.')
            last_backup = None
        else:
            LOG.info('INC is None or INC is 1.')
        # Get backup type, parent_id, and parent_snapshot_url, do Full_backup or Incremental_backup.
        if not last_backup:
            volume_file['backup_type'] = 0
            volume_file['parent_id'] = None
            volume_file['parent_snapshot_url'] = None
        else:
            service_metadata = self.__get_backup_metadata(last_backup)
            if not service_metadata:
                error_msg = convert_error(ErrorCode.EC_COM_INTERNAL_ERROR, "backup service_metadata is none.")
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)
            LOG.info("last_backup %s", last_backup['id'])
            parent_snapshot_id = service_metadata.get('snap_id')
            if parent_snapshot_id is None:
                LOG.info('Last backup is snapshot backup, do full backup.')
                volume_file['backup_type'] = 0
                volume_file['parent_id'] = None
                volume_file['parent_snapshot_url'] = None
            else:
                LOG.info('Last backup is link clone volume backup, do incremental backup.')
                volume_file['backup_type'] = 1
                volume_file['parent_id'] = last_backup['id']
                # Get last backup snapshot_url.
                volume_file['parent_snapshot_url'] = self.__get_snapshot_url(volume_storage_type, vol_info,
                                                                             parent_snapshot_id, True)

        # Get current backup snapshot_url.
        snapshot_url = self.__get_snapshot_url(volume_storage_type, vol_info, snapshot_id, False)
        volume_file['snapshot_url'] = snapshot_url

        # Get snapshot id
        if cst.STORAGE_TYPE_KVMV3 == volume_storage_type:
            # KVM+V3: BACKUPS--service_metadata--snap_id is a real snapshot id of v3 storage
            # eBackup driver will update snapshot_id to BACKUPS--service_metadata--snap_id
            # upgrade from J version to M version, should keep the historical implementation
            volume_file['snapshot_id'] = snapshot_url.split('/').pop().strip()
        else:
            # Others: a openstack snapshot id
            volume_file['snapshot_id'] = snapshot_id

        # Get source volume id
        volume_file['source_volume_id'] = source_volume_id

        # Set default value to reserved parameters
        volume_file['bootable'] = self.__get_none_clone_volume_bootable(source_volume_id)
        volume_file['image_id'] = None
        volume_file['clone_volume_url'] = None

        LOG.info("The volume_file [%s] is constructed", volume_file)
        return volume_file

    def query_redo_backup_process(self, backup):
        try:
            backup_id = backup['id']
            backup = call_db_fun(self.db.backup_get, self.context, backup_id)
            backup_status = backup.get('status')
            backup_fail_reason = backup.get('fail_reason')
            task_type = cst.TASKTYPE_BACKUP
            backup_id = backup['id']
            if backup_status == cst.BACKUP_STATUS_ERROR:
                if backup_fail_reason == cst.EBACKUP_DRIVER_RESTART:
                    self._update_backup(backup_id, status=cst.BACKUP_STATUS_CREATING,
                                        fail_reason='')
            elif backup_status == cst.BACKUP_STATUS_AVALIBLE:
                LOG.info('the backup status is avalible.')
                return
            elif backup_status != cst.BACKUP_STATUS_CREATING:
                error_msg = convert_error(ErrorCode.BD_BACKUP_STATE_NOT_RIGHT, 'backup status is not correct')
                LOG.error('backup status(%s) is not correct' % backup_status)
                raise Exception(error_msg)
            LOG.info('Begin hwsebackup verify, backup_status :%s' % backup_status)

            json_service_metadata = backup.get('service_metadata')
            if json_service_metadata in (None, ''):
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, "service_metadata is null, verify failed.")
                raise exception.InvalidBackup(error_msg)

            metadata = json.loads(json_service_metadata)
            task_id_metadata = metadata.get('ebk_T_I')
            if task_id_metadata in (None, ''):
                msg = "ebk_T_I in service_metadata is null, verify failed."
                LOG.error(msg)
                error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                raise exception.InvalidBackup(error_msg)
            LOG.info('Begin hwsebackup verify, ebk_T_I :%s' % task_id_metadata)

            task_id = ''
            request_id = ''

            if task_id_metadata.startswith('R_'):
                request_id = task_id_metadata[2:]
            else:
                task_id = task_id_metadata

            if task_id in (None, ''):
                task_id = self._get_task_id(task_type, request_id)
                self._change_value_by_keys(metadata, ebk_T_I=str(task_id))
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))

            if is_infiniBand_env and self.__fusionstorage_ip:
                pht_type = 1 if self.__get_connection().get_phy_network_type() == 'ib' else 0
                self._change_value_by_keys(metadata, IB=pht_type)

            while True:
                time.sleep(cst.TIME_SLEEP_30)
                task_infos = self.get_task_info(task_id)
                if not isinstance(task_infos, dict):
                    LOG.error("task : %s failure,failed to get task info" % task_id)
                    break

                task_status, task_progress, task_details = self._get_task_progress_and_details(task_infos, task_type)
                task_speed_kb = task_infos.get("SPEED")
                if task_speed_kb in (None, ''):
                    msg = "SPEED in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)
                task_space_saving_rate = task_infos.get("SPACESAVINGRATE")
                if task_space_saving_rate in (None, ''):
                    msg = "SPACESAVINGRATE in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)

                task_snap_type = task_infos.get("SNAPTYPE")
                if task_snap_type in (None, ''):
                    msg = "SNAPTYPE in task_info is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)

                MbToKb = cst.Mi / cst.Ki
                task_speed_mb = float(task_speed_kb) / MbToKb

                dict_container_info = {}
                json_container = backup.get('container')
                if json_container in (None, ''):
                    dict_container_info["AT"] = round(task_speed_mb, 1)
                    dict_container_info["SS"] = int(task_space_saving_rate)
                else:
                    if -1 == json_container.find(":"):  # J old data
                        dict_container_info["AT"] = round(task_speed_mb, 1)
                        dict_container_info["SS"] = int(task_space_saving_rate)
                        dict_container_info["SNAPID"] = json_container
                    else:
                        dict_container_info = eval(json_container)
                        at_metadata = dict_container_info.get("AT")
                        if at_metadata in (None, ''):
                            msg = "AT in container is null, verify failed."
                            LOG.error(msg)
                            error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                            raise exception.InvalidBackup(error_msg)
                        else:
                            dict_container_info["AT"] = round(task_speed_mb, 1)
                        ss_metadata = dict_container_info.get("SS")
                        if ss_metadata in (None, ''):
                            msg = "SS in container is null, verify failed."
                            LOG.error(msg)
                            error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                            raise exception.InvalidBackup(error_msg)
                        else:
                            dict_container_info["SS"] = int(task_space_saving_rate)

                self._update_backup(backup_id, container=json.dumps(dict_container_info))
                if task_snap_type is not None:  # old version
                    backup_mode = self._change_openstack_mode(task_snap_type)
                    self._change_value_by_keys(metadata, Type=backup_mode)
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))

                LOG.info('verify a task progress is:%s' % task_progress)
                # backup failed
                if task_status == cst.TASK_STATUS_FAILED:
                    msg = "task failed, eBackup task id is:%s" % task_id
                    fail_reason_str = self._get_task_detail(task_details, msg)
                    LOG.error("Job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                    raise exception.InvalidBackup(fail_reason_str)

                self._change_value_by_keys(metadata, progress=str(task_progress))
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))

                # backup completed
                if task_status == cst.TASK_STATUS_COMPLETED:
                    image_id = self._get_new_snap_uuid(task_id)
                    if image_id is None:
                        # if image id is null, it means backup failed.
                        error_msg = convert_error(ErrorCode.IMAGE_NON_EXISTENT, 'get new image id is None')
                        LOG.error(error_msg)
                        raise exception.InvalidBackup(error_msg)

                    volumeid = backup["volume_id"]
                    volume = call_db_fun(self.db.volume_get, self.context, volumeid)
                    # get volume backup size (total size)
                    domain_type = cst.DOMAIN_TYPE_BACKUP
                    if 'IT' in metadata:
                        domain_type = metadata['IT']
                    volume_backup_size, update, ad_flag = self._get_total_chain_size(self._get_cascaded_volume_id(backup),
                                                                                     domain_type,
                                                                                     metadata.get("ST"),
                                                                                     backup)
                    self._remove_keys(metadata, "backupurl_tmp")
                    if ad_flag:
                        self._change_value_by_keys(metadata, AD=1)
                    self._change_value_by_keys(metadata, backupurl=str(image_id), progress='0', ebk_T_I="")
                    self._update_backup(backup_id, object_count=volume_backup_size,
                                        service_metadata=json.dumps(metadata),
                                        availability_zone=volume["availability_zone"],
                                        status=cst.BACKUP_STATUS_AVALIBLE,
                                        service='cinder.backup.drivers.hwsebackup')
                    LOG.info("The volume_file [%s] is constructed", volumeid)
                    if ('snapshot_id' in backup.keys()) and (backup.get('snapshot_id') is not None):
                        call_db_fun(self.db.snapshot_update, self.context, backup.get('snapshot_id'),
                                    {'status': 'available'})
                    else:
                        call_db_fun(self.db.volume_update, self.context, volumeid, {'status': 'available'})
                    LOG.info("update ok!")
                    break

        except Exception as msg:
            with excutils.save_and_reraise_exception():
                LOG.error('exception is:%s' % msg)
                if str(msg) == str(cst.GOOD_EXCEPTION):
                    self._update_backup(backup_id, status=cst.BACKUP_STATUS_CREATING,
                                        fail_reason='')
                    LOG.info("update creating")
                else:
                    json_service_metadata = backup.get('service_metadata')
                    metadata = json.loads(json_service_metadata)
                    self._change_value_by_keys(metadata, progress='0')
                    self._update_backup(backup_id, status=cst.BACKUP_STATUS_ERROR,
                                        service_metadata=json.dumps(metadata),
                                        fail_reason=str(msg))
                    LOG.info("update error")
        LOG.info("verify backup(%s) finished." % backup_id)

    def query_redo_restore_process(self, backup, task_id, task_type):
        """Restore the given volume backup from eBackup server.

        :param backup: backup information for volume
        :param task_id: task id
        :raises InvalidBackup
        """

        try:

            backup_id = backup['id']
            if len(backup_id) != cst.LENGTH_LIMIT_36:
                msg = ('backup id length does not comply, error length:%d' % len(backup_id))
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise Exception(error_msg)

            LOG.info('Begin hwsebackup restore, backup id:%s' % backup_id)
            # update task status
            json_service_metadata = backup['service_metadata']
            if json_service_metadata in (None, ''):
                msg = 'service metadata is null, restore can not begin.'
                error_msg = convert_error(ErrorCode.BD_BACKUP_PARAM_WRONG, msg)
                LOG.error(error_msg)
                raise exception.InvalidBackup(error_msg)

            metadata = json.loads(json_service_metadata)
            while True:
                time.sleep(cst.TIME_SLEEP_30)
                task_infos = self.get_task_info(task_id)
                if not isinstance(task_infos, dict):
                    LOG.error("task : %s failure,failed to get task info" % str(task_id))
                    break

                task_status, task_progress, task_details = self._get_task_progress_and_details(task_infos,
                                                                                               task_type)
                LOG.info('restore task progress is:%s' % task_progress)
                if task_status == cst.TASK_STATUS_FAILED:
                    msg = ("eBackup Restore VM disk job failed, job id is:" + str(task_id))
                    fail_reason_str = self._get_task_detail(task_details, msg)
                    LOG.error("Restore job(%s) failed, reason:%s" % (str(task_id), fail_reason_str))
                    raise exception.InvalidBackup(fail_reason_str)

                # update task progress.
                if task_progress == '0':
                    task_progress = '1'
                self._change_value_by_keys(metadata, progress=str(task_progress))
                self._update_backup(backup_id, service_metadata=json.dumps(metadata))

                # speed
                ret_status, task_speed = self._get_task_details_speed(task_infos)
                if ret_status is False:
                    msg = "SPEED in restore is null, verify failed."
                    LOG.error(msg)
                    error_msg = self._convert_error(cst.VERIFY_TASK_NOT_FOUND, msg)
                    raise exception.InvalidBackup(error_msg)

                if -1 != task_speed:
                    self._change_value_by_keys(metadata, AT=task_speed)
                    self._update_backup(backup_id, service_metadata=json.dumps(metadata))

                if task_status == cst.TASK_STATUS_COMPLETED:
                    self._change_value_by_keys(metadata, progress='0', ebk_T_I='')
                    self._update_backup(backup_id, service_metadata=json.dumps(metadata),
                                        status=cst.BACKUP_STATUS_AVALIBLE)
                    target_volume_id = backup.restore_volume_id
                    LOG.info("The target_volume_id is [%s] ", target_volume_id)
                    call_db_fun(self.db.volume_update, self.context, target_volume_id, {'status': 'available'})
                    break
            LOG.info('%s:End hwsebackup restore.' % task_id)

        except Exception as msg:
            fail_reason_str = ("%s" % msg)
            LOG.error(fail_reason_str)
            self._change_value_by_keys(metadata, progress='0')
            self._update_backup(backup_id, service_metadata=json.dumps(metadata), fail_reason=fail_reason_str)
            raise exception.InvalidBackup(msg)

    def vendor_manage_backup(self, backup, host, availability_zone):
        service_meta = backup['service_metadata']
        LOG.info('pre migrate %s,%s,%s,%s' % (backup['id'], backup['host'], backup['availability_zone'], service_meta))
        if backup['status'] in (cst.BACKUP_STATUS_AVALIBLE, cst.BACKUP_STATUS_RESTORING):
            sm_json = json.loads(service_meta)
            self._change_value_by_keys(sm_json, migrate=cst.XEN_TO_KVM)
            service_meta = json.dumps(sm_json)
        self._update_backup(backup['id'], service_metadata=service_meta, host=host,
                            availability_zone=availability_zone)
        LOG.info(
            'migrate backup %s to %s,%s,%s' % (backup['id'], host, availability_zone, service_meta))
        return

    def vendor_unmanage_backup(self, backup, host, availability_zone):
        service_meta = backup['service_metadata']
        LOG.info('pre rollback %s,%s,%s,%s' % (backup['id'], backup['host'], backup['availability_zone'], service_meta))
        if backup['status'] in (cst.BACKUP_STATUS_AVALIBLE, cst.BACKUP_STATUS_RESTORING):
            sm_json = json.loads(service_meta)
            if 'migrate' in sm_json.keys():
                del sm_json['migrate']
            service_meta = json.dumps(sm_json)
        self._update_backup(backup['id'], service_metadata=service_meta, host=host,
                            availability_zone=availability_zone)
        LOG.info(
            'rollback backup %s to %s,%s,%s' % (backup['id'], host, availability_zone, service_meta))
        return


class ExpensesEvent(object):
    def __init__(self, db, typed):
        self._type = typed
        self.db = db

    def get_backup_private_tags(self, context, backup_id):
        if not hasattr(self.db, 'backup_tags_list_get') or not hasattr(self.db, 'backup_tags_get'):
            LOG.info("tag interfaces aren't available, no tag export")
            return {}
        backup_list = [backup_id]
        public_tags = self.db.backup_tags_list_get(context, backup_list)
        public_tags_dict = {}
        for tag in public_tags:
            if not tag['is_public']:
                continue
            public_tags_dict[tag['key']] = tag['value']
        tags = self.db.backup_tags_get(context, backup_id)
        private_tag = {k: tags[k] for k in set(tags) - set(public_tags_dict)}
        LOG.info("private_tag: %s", private_tag)
        return private_tag

    def send_expenses_event(self, context, current_backup, charge_backup, successd):
        """
        :param context: context
        :param current_backup: current oprate backup
        :param charge_backup: backup used to update cost
        :param successd: is opration complete
        :return: none
        """
        if charge_backup is None:
            charge_backup = dict()
        if 'is_csd_layer' not in CINDER_CONF:
            LOG.info("current node don't have is_csd_layer in conf")
            return
        backup = current_backup
        display_desc = current_backup['display_description']
        try:
            json_display_desc = Utils.transfer_json_to_dict(display_desc)
            st = json_display_desc.get('ST')
            if st:
                resource_type = 'csbs'
            else:
                resource_type = 'vbs'
        except Exception as msg:
            LOG.info("get ST from description failed, invalid json format. %s" % msg)
            resource_type = 'vbs'
        if successd:
            backup['status'] = 'available'
        else:
            backup['status'] = 'error'
        if self._type in ['backup', 'copy']:
            event_name = 'backup.create.end'
        elif self._type in ['delete']:
            event_name = 'backup.delete.end'
        else:
            return
        backup['servive_metadata'] = charge_backup.get('service_metadata', '')
        backup['object_count'] = charge_backup.get('object_count', 0)
        backup['size'] = charge_backup.get('size', 0)
        if CINDER_CONF.is_csd_layer:
            LOG.info("current layer is cascaded, send event to cascading!")
            charge_event_list = list()
            chg_e = dict()
            bid = backup['display_name']
            chg_e['backup_id'] = bid[bid.index('@') + 1:]
            chg_e['size'] = backup['size']
            chg_e['object_count'] = backup['object_count']
            chg_e['resource_id'] = bid[bid.index('@') + 1:]
            chg_e['resource_type'] = resource_type
            chg_e['volume_id'] = self._get_cascade_layer_volume_id(
                context, backup['volume_id'])
            chg_e['event_name'] = event_name
            chg_e['description'] = charge_backup.get('display_description', '')
            chg_e['project_id'] = backup['project_id']
            chg_e['host'] = backup['host']
            chg_e['status'] = backup['status']
            chg_e['backup_tags'] = json.dumps(self.get_backup_private_tags(context, backup['id']))
            chg_e['service_metadata'] = backup['servive_metadata']
            charge_event_list.append(copy.deepcopy(chg_e))
            if hasattr(volume_utils, 'notify_backup_csding'):
                volume_utils.notify_backup_csding([], charge_event_list, [])
                LOG.info(
                    "notified, type:%s, event_name: %s,id %s,service_metadata: %s, count:%s, desc %s, tags %s" % (
                        resource_type, event_name, backup['id'], backup['service_metadata'], backup['object_count'],
                        chg_e['description'], chg_e['backup_tags']))
            else:
                LOG.info("not found function notify_backup_csding for notify backup %s event" % current_backup['id'])
        else:
            backup['display_description'] = charge_backup.get('display_description', '')
            LOG.info("current layer is cascading,insert event to db!")
            if hasattr(self.db, 'backup_event_create'):
                call_db_fun(self.db.backup_event_create, context, backup, event_name, resource_type)
                LOG.info(
                    "add event seccessed, type:%s, event_name: %s,id %s,service_metadata: %s, count:%s, desc %s" % (
                        resource_type, event_name, backup['id'], backup['service_metadata'], backup['object_count'],
                        backup['display_description']))
            else:
                LOG.info("not found function db.backup_event_create for insert backup %s event" % current_backup['id'])

    def _get_cascade_layer_volume_id(self, context, volume_id):
        try:
            volume = call_db_fun(self.db.volume_get, context, volume_id)
            display_name = volume.get('display_name', '')
            if 'volume@' in display_name:
                return display_name.split('@')[-1]
        except Exception as ex:
            LOG.exception('Failed to get cascading volume id: {}.'.format(ex))
        return volume_id

    @staticmethod
    def sync_backup_to_cascading(backup_name, key, value):
        if 'is_csd_layer' not in CINDER_CONF:
            LOG.info("current node don't have is_csd_layer in conf")
            return
        backup_list = list()
        bkp = dict()
        bkp[key] = value
        if CINDER_CONF.is_csd_layer:
            bkp['id'] = backup_name[backup_name.index('@') + 1:]
            LOG.info("current layer is cascaded, sync backup to cascading!")
            backup_list.append(bkp)
            if hasattr(volume_utils, 'notify_backup_csding'):
                volume_utils.notify_backup_csding(backup_list, [], [])
                LOG.info(
                    "notified, backup:%s, key: %s,value %s" % (backup_name, key, value))
            else:
                LOG.info("not found function notify_backup_csding for notify backup %s event" % backup_name)
        else:
            LOG.info("current layer is cascading, do not sync backup")

    @staticmethod
    def sync_tags_to_cascading(backup_name, tags):
        if 'is_csd_layer' not in CINDER_CONF:
            LOG.info("current node don't have is_csd_layer in conf")
            return

        backup_list = list()
        bkp = dict()
        bkp['is_public'] = False
        bkp['tags'] = tags
        if CINDER_CONF.is_csd_layer:
            bkp['backup_id'] = backup_name[backup_name.index('@') + 1:]
            LOG.info("current layer is cascaded, sync tags to cascading!")
            backup_list.append(bkp)
            if hasattr(volume_utils, 'notify_backup_csding'):
                volume_utils.notify_backup_csding([], [], backup_list)
                LOG.info(
                    "notified, backup:%s, tags %s" % (backup_name, json.dumps(tags)))
            else:
                LOG.info("not found function notify_backup_csding for notify tags %s event" % backup_name)
        else:
            LOG.info("current layer is cascading, do not sync tags")


def get_backup_driver(context):
    return EbackupDriver(context)
